// *==============================================================================================
// *
// *   1636RR4U.v - 16M-BIT CMOS Voltage 3.3V Flash Memory
// *
// *           2021 Milandr IC Design Center
// *----------------------------------------------------------------------------------------------
// * Environment  : Cadence NC-Verilog
// * Creation Date: Date: 2021/17/03 14:40:56
// * Version      : Revision: 1.0
// * Description  : There is only one module in this file
// *                module K1636RR4U-> behavior model for the 16M-Bit flash
// *----------------------------------------------------------------------------------------------
// * Note 1:model can load initial flash data from file when parameter PRELOAD in "1" state and
// *        memory_file.txt: initial flash data file name; 
// *        when parameter PRELOAD in "0" state, initial flash data is "xxxx".
// * Note 2:model can pre-initialize flash data to erased when parameter PRE_ERASE in "1" state;
// *        Parameter PRELOAD has a higher priority;
// * Note 3:power setup time is Tvcs = 100us, so after power up, chip can be enable;
// *        user can modify this parameter for behavioural simulation;
// *        true hardware value for this parameter is 4ms;
// * Note 4:console log with important and timing violation messages can be enabled with parameter
// *        CONSOLE_LOG in "1" state; by default enabled
// * Note 5:extra log with other messages can be enabled with parameter EXTRA_LOG in "1" state;
// *        by default disabled
// * Note 6:if you have any question and suggestion, please visit following address:
// *                        https://support.milandr.ru/ask/
// *----------------------------------------------------------------------------------------------
// * History
// * Date  |  Version   Description
// *==============================================================================================
`timescale 1ns / 1ps

module K1636RR4U (
  A,
  D,
  nCE,
	nWE,
  nOE,
  A9_HV,
  OE_HV,
  SEL_SPI,
  SCK,
  SI,
  SO,
  MRST,
  STROBE,
  TCK,
  TDI
  ); 

parameter	Tvcs = 100_000;	// Ucc setup time in ns

parameter PRELOAD = 0;										  // Pre-load enable/disable 
parameter PRE_ERASE = 0;		  			        // Pre-initialization to erased enable/disable
parameter MEMORY_FILE = "memory_file.txt";  // Memory pre-load file define

parameter CONSOLE_LOG = 1; // To enable log for console set this parameter to 1
                           // by default enabled
parameter EXTRA_LOG = 0; // To enable detailed log set this parameter to 1
                         // by default disabled and only critical messages will be logged

//========================================================================
// Array constants table  
//========================================================================
parameter A_MSB     = 21;
parameter A_SEC_MSB = 18;
parameter D_MSB     = 8;

parameter numWLP  = 8;
parameter numWL   = 1024;
parameter numMux  = 256;
parameter numSect = 8;
parameter numPage = numWL / numWLP;

parameter sizePage   = numWLP * numMux;
parameter sizeSector = numWL * numMux;
parameter sizeMem    = numSect * sizeSector;

wire [7:0] id_device      = 8'hC8;
wire [7:0] id_manufactory = 8'h01;

integer i, j, k;

//========================================================================
// Timing units table  
//========================================================================
parameter ns_unit =  1;
parameter us_unit =  1000;
parameter ms_unit =  1000 * 1000;
parameter sec_unit = 1000 * 1000 * 1000;

//========================================================================
// Global timing table  
//========================================================================
parameter Ter_chip = 3; // Chip Erase time in sec
parameter Ter_sec  = 220; // Sector Erase time in ms
parameter Ter_pg   = 200; // Page Erase time in ms
parameter Twr      = 200; // Program byte in us
parameter Tw_sec_load = 50; // Timeout window for sector load in us
parameter Tw_we_pr = 100; // Sector protect pulse width in us
parameter Tw_pr_we = 40; // Sector unprotect pulse width in ms
parameter Ts_oe_pr = 4; // OE_HV setup time for protect/unprotect in us
parameter Ts_a9_pr = 4; // A9_HV setup time for protect/unprotect in us

//========================================================================
// Timing table for PARALLEL I/F  
//========================================================================
specify

  specparam Tv_oe_d   = 0; // Toh
  specparam Ta_a      = 55; // Tacc
  specparam Ta_ce     = 65; // Tacc
  specparam Ta_oe     = 55; // Tacc
  specparam Tpdz_oe_d = 16; // Tdf
	
  specparam Ts_a = 0; // Tas
  specparam Th_a = 45; // Tah

  specparam Ts_d = 30; // Tds
  specparam Th_d = 15; // Tdh

  specparam Ts_cel_wel = 0; // Tcs
  specparam Ts_oeh_wel = 0; // Toes
  specparam Th_weh_cel = 0; // Tch

  specparam Ts_wel_cel = 0; // Tws
  specparam Ts_oeh_cel = 0; // Toes
  specparam Th_ceh_wel = 0; // Twh

  specparam Th_oeh_weh = 13; // Toeh
 
  specparam Twph   		 = 35; // Twph
  specparam Twpl   		 = 30; // Twp
  specparam Tcph   		 = 35; // Tcph
  specparam Tcpl   		 = 30; // Tcp

  specparam Tcyr   		 = 70; // Trc
  specparam Tcyw   		 = 70; // Twc

endspecify

//========================================================================
// Timing table for SERIAL I/F  
//========================================================================
specify

  specparam T_tck_min    = 20;

  specparam T_tck_er_max = 640;
  specparam T_tck_er_min = 380;

  specparam T_tck_wr_max = 240;
  specparam T_tck_wr_min = 140;

  specparam Ts_tck_tdi  = 5;     
  specparam Th_tdi_tck   = 0; 
  specparam Tpzd_tck_d   = 10;
  specparam Tpdz_tck_d   = 10; 

  specparam Ta_tck       = 35; 

endspecify

//========================================================================
// Timing table for SPI I/F  
//========================================================================
parameter Treset = 30; // in us

specify

  specparam T_sck_rd1 = 33.3;
  specparam T_sck_rd2 = 66.6;
  specparam T_sck_min = 20;

  specparam Twh_sck = 10; // Tclkh
  specparam Twl_sck = 10; // Tclkl

  specparam Ts1_ce = 10; // Tcsls
  specparam Ts2_ce = 10; // Tcshs
  specparam Th1_ce = 5; // Tcslh
  specparam Th2_ce = 5; // Tcshh

  specparam Twh1_ce = 1000; // Tcsh
  specparam Twh2_ce = 50; // Tcsh

  specparam Ts_sck_si = 2 ; // Tds     
  specparam Th_si_sck = 1 ; // Tdh

  specparam Tpzd_sck_d = 10; // Tv
  specparam Tpdz_sck_d = 10; // Tdis

endspecify

/*----------------------------------------------------------------------*/
/* I/O Ports                                                            */
/*----------------------------------------------------------------------*/
input [A_MSB-1:0] A; 
inout [D_MSB-1:0] D;
input             nCE;
input             nWE;
input             nOE;

input tri0        A9_HV;
input tri0        OE_HV;

input tri0        SEL_SPI;
input tri0        SCK;
input tri0        SI;
output            SO;

input tri0        MRST;	
input tri0        STROBE;
input tri0        TCK;
input tri0        TDI;

/*----------------------------------------------------------------------*/
/* I/O Lines                                                            */
/*----------------------------------------------------------------------*/
wire [A_MSB-1:0]  A_IN; 
wire [D_MSB-1:0]  D_IN;
reg  [D_MSB-1:0]  D_OUT;
wire              nCE_IN;
wire              nWE_IN;
wire              nOE_IN;

wire              A9_HV_IN;
wire              OE_HV_IN;

wire              SEL_SPI_IN;
wire              SCK_IN;
wire              SI_IN;
reg               SO_OUT;

wire              MRST_IN;	
wire              STROBE_IN;
wire              TCK_IN;
wire              TDI_IN;

/*----------------------------------------------------------------------*/
/* SPI I/F interconnect                                                 */
/*----------------------------------------------------------------------*/
wire              SEL_SPI_work;
reg  [5:0]  		  bit_counter;
reg  [5:0]  		  bit_counter2;
reg  [7:0]  		  opcode;
reg               execute_opcode;
reg               reset_request;
reg               incycle;

reg  [A_MSB-1:0]  A_SPI;
reg  [7:0]        D_PROG_SPI;
reg  [6:0]        data_shift;

reg        			  SPRL;
reg        			  RSTE;
reg       			  EPE;
wire [1:0] 			  SWP;
reg               WEL;
reg               RDY_BSY_SPI;

reg [numSect-1:0] Sector_Protect_Status_SPI; // Volatile

/*----------------------------------------------------------------------*/
/* Serial I/F interconnect                                              */
/*----------------------------------------------------------------------*/
reg               SEL_SERIAL;
wire              SEL_SERIAL_work;
wire              MRST_work;
wire              TDI_IN_work;
reg               TDI_OUT;

reg [A_MSB-1:0]   A_SERIAL;
reg [D_MSB-1:0]   DIN_SERIAL;

reg [1:0]         State_current;
reg [1:0]         State_previous;

reg [10:0]        command;
reg [16:0]        address;
reg [15:0]        data;  
reg               RDY_BSY_SERIAL;
reg [7:0]         seq;

reg [5:0]         count_bit;
reg               rep_eat; 
reg               read_flag;
wire              MRST_STROBE;

integer cnt_tck;

/*----------------------------------------------------------------------*/
/* Parallel I/F interconnect                                            */
/*----------------------------------------------------------------------*/
wire            SEL_PAR;
wire            RSTN_PAR;
wire            reset_protect;
wire            event_r; // read cycle on nCE-nWE-nOE
wire            event_d; // write cycle on nCE-nWE-nOE

reg [7:0]       FSM_state_PAR;
reg [2:0]       CMD_cycle_PAR;
reg             was_zero_event;
reg             bypass;

reg             DQ2;
reg             DQ3;
reg             DQ5_write;
reg             DQ5_erase;
reg             DQ6;

reg [A_MSB-1:0] A_REG;
reg [A_MSB-1:0] A_RD_PAR;
reg [A_MSB-1:0] A_PROG_PAR;
reg [D_MSB-1:0] D_PROG_PAR;
reg [D_MSB-1:0] Status_REG;

reg [numSect-1:0]         Sector_Protect_Status_PAR; // nonVolatile
reg [clog2(numSect)-1:0]  sector_buf [numSect-1:0];
reg [clog2(numSect):0]    sector_pointer;

wire seq1;
wire seq2;
wire seq_read_reset;
wire seq_autoselect;
wire seq_byte_program;
wire seq_erase;
wire seq_chip_erase;
wire seq_bypass;

/*----------------------------------------------------------------------*/
/* Internal operations routines and constants                           */
/*----------------------------------------------------------------------*/
time WR_time        = Twr      * us_unit;
time ER_Page_time   = Ter_pg   * ms_unit;
time ER_Sector_time = Ter_sec  * ms_unit;
time ER_Chip_time   = Ter_chip * sec_unit;
time SPR_time       = Tw_we_pr * us_unit;
time SUPR_time      = Tw_pr_we * ms_unit;
time Timeout_Window_time = Tw_sec_load * us_unit;

event Event_WR;
event Event_ER_Page;
event Event_ER_Sector;
event Event_ER_Chip;
event Event_RD;
event Event_OUT;
event Event_SPR;
event Event_SUPR;
event Event_ER_Sector_tst;
event Event_WR_tst;

reg Process_WR;             // a flag to indicate in Program state
reg Process_ER_Page;        // a flag to indicate in Erase Page state
reg Process_ER_Sector;      // a flag to indicate in Erase Sector state
reg Process_ER_Chip;        // a flag to indicate in Erase Chip state
reg Process_SPR;            // a flag to indicate Sector Protection Set state
reg Process_SUPR;           // a flag to indicate Sector Protection Unset state
reg Load_ER_Sector;         // a flag to indicate Sector Erase Load state   
reg Process_ER_Sector_tst;  // a flag to indicate in Erase Sector test state
reg Process_WR_tst;         // a flag to indicate in Program test state

wire Flag_SPR_Verify; // verify sector protect using OE_VID and A9_VID
wire Flag_SPR;        // set sector protect using OE_VID and A9_VID
wire Flag_SUPR;       // set sector unprotect using OE_VID and A9_VID

/*----------------------------------------------------------------------*/
/* Power on reset control                                               */
/*----------------------------------------------------------------------*/
time power_on_reset = Tvcs;
reg RSTN;

initial begin
  RSTN = 1'b0;
  #(power_on_reset);
  RSTN = 1'b1;
  if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "Model wakes up at %0dns \n", Tvcs);
end
		
/*----------------------------------------------------------------------*/
/* I/O bus opearation                                                   */
/*----------------------------------------------------------------------*/
time Tglitch = 5; // in ns
time Tio = 0.001; // in ns

reg nCE_temp, nCE_current;
reg nWE_temp, nWE_current;
reg nOE_temp, nOE_current;

wire [A_MSB-1:0] A_DLY;
wire [D_MSB-1:0] D_DLY;
wire nCE_dly;
wire nWE_dly;
wire nOE_dly;

wire nCE_GF; // glitch-free nCE
wire nWE_GF; // glitch-free nWE
wire nOE_GF; // glitch-free nOE

event nCE_switch;
event nWE_switch;
event nOE_switch;

assign #(Tglitch) A_DLY = A;
assign #(Tglitch) D_DLY = D;
assign #(Tglitch) nCE_dly = nCE;
assign #(Tglitch) nWE_dly = nWE;
assign #(Tglitch) nOE_dly = nOE;

always @ (nCE) if (nCE_dly == nCE) if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "Warning! Glitch on nCE <5ns was suppressed \n"); 
always @ (nWE) if (nWE_dly == nWE) if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "Warning! Glitch on nWE <5ns was suppressed \n"); 
always @ (nOE) if (nOE_dly == nOE) if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "Warning! Glitch on nOE <5ns was suppressed \n"); 

always @ (nCE) begin
disable nCE_glitch_check;
->nCE_switch;
nCE_temp = nCE;
end

always @ (nWE) begin
disable nWE_glitch_check;
->nWE_switch;
nWE_temp = nWE;
end

always @ (nOE) begin
disable nOE_glitch_check;
->nOE_switch;
nOE_temp = nOE;
end

always @ (nCE_switch) begin: nCE_glitch_check
#(Tglitch);
nCE_current <= nCE_temp;
end

always @ (nWE_switch) begin: nWE_glitch_check
#(Tglitch);
nWE_current <= nWE_temp;
end

always @ (nOE_switch) begin: nOE_glitch_check
#(Tglitch);
nOE_current <= nOE_temp;
end

assign #Tio A_IN       = A;
assign #Tio D_IN       = D;
assign #Tio nCE_IN     = nCE;
assign #Tio nOE_IN     = nOE;
assign #Tio nWE_IN     = nWE;
assign #Tio OE_HV_IN   = OE_HV;
assign #Tio A9_HV_IN   = A9_HV;
assign #Tio SEL_SPI_IN = SEL_SPI;
assign #Tio SI_IN      = SI;
assign #Tio SCK_IN     = SCK;
assign #Tio TDI_IN     = TDI;
assign #Tio MRST_IN    = MRST;
assign #Tio STROBE_IN  = STROBE;
assign #Tio TCK_IN     = TCK;

assign nCE_GF = nCE_current;
assign nOE_GF = nOE_current;
assign nWE_GF = nWE_current;

/*----------------------------------------------------------------------*/
/* Output control                                                       */
/*----------------------------------------------------------------------*/
wire OE_D;
reg  OE_SO;
reg  OE_TDO;

assign D   = (RSTN && !OE_D)   ? 8'bz : D_OUT;
assign TDI = (RSTN && !OE_TDO) ? 1'bz : TDI_OUT;
assign SO  = (RSTN && !OE_SO)  ? 1'bz : SO_OUT;
	
//==============================================================================================================================================
//==============================================================================================================================================
// FLASH ARRAY and sector architecture
//==============================================================================================================================================
//==============================================================================================================================================
reg [D_MSB-1:0]   ARRAY [0:sizeMem-1]; // Flash Array

reg mem_initialized;	

initial begin // memory initialization
	if (PRELOAD) begin // pre-load, if needed	
    $readmemh(MEMORY_FILE, ARRAY);
    if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "Successful pre-load memory array \n");
    mem_initialized = 1;
  end else if (PRE_ERASE) begin
    for (j = 0; j < sizeMem; j = j + 1) begin // pre-initiazliation to erased  	
      ARRAY[j] = {D_MSB{1'b1}};    
    end
    if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "Successful pre-initiazliation memory array to erased \n");
    mem_initialized = 1;
  end else begin
    for (j = 0; j < sizeMem; j = j + 1) begin // by default start with undefine state  	
      ARRAY[j] = {D_MSB{1'bx}};    
    end
    mem_initialized = 0;
  end
end

function integer clog2 (input integer num); // this function calculates ceil(log2(num))
begin    
  num = num - 1;                            
  for (clog2 = 0; num > 0; clog2 = clog2 + 1) num = num >> 1;
end
endfunction

//==============================================================================================================================================
//==============================================================================================================================================
// PARALLEL I/F
//==============================================================================================================================================
//==============================================================================================================================================
`define State_read_reset      8'b00000001 
`define State_autoselect      8'b00000010 
`define State_byte_program    8'b00000100 
`define State_chip_erase      8'b00001000 
`define State_sector_erase    8'b00010000 
`define State_page_erase      8'b00100000
`define State_erase           8'b01000000 
`define Start_up_byte_program 8'b10000000 

`define Null_cycle            3'h0
`define First_cycle           3'h1
`define Second_cycle          3'h2
`define Third_cycle           3'h3
`define Fourth_cycle          3'h4
`define Fifth_cycle           3'h5
`define Sixth_cycle           3'h6

/*----------------------------------------------------------------------*/
/* Specify the timing                                                   */
/*----------------------------------------------------------------------*/
wire event_d_t = !nCE_IN && nOE_IN && !nWE_IN;

reg notify_Ts_a, viol_Ts_a;
reg notify_Ts_d, viol_Ts_d;
reg notify_Th_a, viol_Th_a;
reg notify_Th_d, viol_Th_d;

reg notify_Ts_cel_wel, viol_Ts_cel_wel;
reg notify_Ts_oeh_wel, viol_Ts_oeh_wel;
reg notify_Ts_wel_cel, viol_Ts_wel_cel;
reg notify_Ts_oeh_cel, viol_Ts_oeh_cel;

reg notify_Th_weh_cel, viol_Th_weh_cel;
reg notify_Th_ceh_wel, viol_Th_ceh_wel;

reg notify_Th_oeh_weh, viol_Th_oeh_weh;

reg notify_Twph, viol_Twph;
reg notify_Twpl, viol_Twpl;
reg notify_Tcph, viol_Tcph;
reg notify_Tcpl, viol_Tcpl;

reg notify_Tcyr, viol_Tcyr;
reg notify_Tcyw, viol_Tcyw;

reg notify_Ts_oe_pr, viol_Ts_oe_pr;
reg notify_Ts_a9_pr, viol_Ts_a9_pr;

reg ctrl_nCE;
reg ctrl_nWE;

wire check_Ts_a = RSTN && SEL_PAR;
wire check_Ts_d = RSTN && SEL_PAR;
wire check_Th_a = RSTN && SEL_PAR;
wire check_Th_d = RSTN && SEL_PAR;

wire check_Ts_cel_wel = RSTN && SEL_PAR;
wire check_Ts_oeh_wel = RSTN && ctrl_nWE && SEL_PAR;
wire check_Ts_wel_cel = RSTN && SEL_PAR;
wire check_Ts_oeh_cel = RSTN && ctrl_nWE && SEL_PAR;

wire check_Th_weh_cel = RSTN && SEL_PAR && ctrl_nWE;
wire check_Th_ceh_wel = RSTN && SEL_PAR && ctrl_nCE;

wire check_Th_oeh_weh = RSTN && SEL_PAR && ctrl_nWE;

wire check_Twp = RSTN && SEL_PAR && ctrl_nWE && was_zero_event;
wire check_Tcp = RSTN && SEL_PAR && ctrl_nCE && was_zero_event;

wire check_Tcyr = RSTN && SEL_PAR && ~nCE_IN && ~nOE_IN && nWE_IN;
wire check_Tcyw = RSTN && SEL_PAR;

wire check_Ts_oe_pr = RSTN && SEL_PAR;
wire check_Ts_a9_pr = RSTN && SEL_PAR;

time Tneg_nCE;
time Tneg_nWE;
time Tpos_nCE;
time Tpos_nWE;
time Tpos_event_d;
time Tch_A_w;
time Tch_A_r;
time Tpos_OE_HV;
time Tpos_A9_HV;

initial begin
  notify_Ts_a = 0;
  notify_Ts_d = 0;
  notify_Th_a = 0;
  notify_Th_d = 0;

  notify_Ts_cel_wel = 0;
  notify_Ts_oeh_wel = 0;
  notify_Ts_wel_cel = 0;
  notify_Ts_oeh_cel = 0;

  notify_Th_weh_cel = 0;
  notify_Th_ceh_wel = 0;

  notify_Th_oeh_weh = 0;

  notify_Twph = 0;
  notify_Twpl = 0;
  notify_Tcph = 0;
  notify_Tcpl = 0;

  notify_Tcyr = 0;
  notify_Tcyw = 0;

  notify_Ts_oe_pr = 0;
  notify_Ts_a9_pr = 0;

  viol_Ts_a = 0;
  viol_Ts_d = 0;
  viol_Th_a = 0;
  viol_Th_d = 0;

  viol_Ts_cel_wel = 0;
  viol_Ts_oeh_wel = 0;
  viol_Ts_wel_cel = 0;
  viol_Ts_oeh_cel = 0;

  viol_Th_weh_cel = 0;
  viol_Th_ceh_wel = 0;

  viol_Th_oeh_weh = 0;

  viol_Twph = 0;
  viol_Twpl = 0;
  viol_Tcph = 0;
  viol_Tcpl = 0;

  viol_Tcyr = 0;
  viol_Tcyw = 0;

  viol_Ts_oe_pr = 0;
  viol_Ts_a9_pr = 0;

  ctrl_nCE = 0;
  ctrl_nWE = 0;

  Tneg_nCE = 0;
  Tneg_nWE = 0;
  Tpos_nCE = 0;
  Tpos_nWE = 0;
  Tpos_event_d = 0;
  Tch_A_w = 0;
  Tch_A_r = 0;
  Tpos_OE_HV = 0;
  Tpos_A9_HV = 0;
end

// Notify section
always @ (notify_Th_a) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun address hold time Th_a=%0dns \n", Th_a);
  viol_Th_a = 1;
end

always @ (notify_Th_d) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun data hold time Th_d=%0dns \n", Th_d);
  viol_Th_d = 1;
end

always @ (notify_Ts_a) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun address setup time Ts_a=%0dns \n", Ts_a);
  viol_Ts_a = 1;
end

always @ (notify_Ts_d) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun data setup time Ts_d=%0dns \n", Ts_d);
  viol_Ts_d = 1;
end

always @ (notify_Twph) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun positive pulse width time Twph=%0dns \n", Twph);
  viol_Twph = 1;
end

always @ (notify_Twpl) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun negative pulse width time Twpl=%0dns \n", Twpl);
  viol_Twpl = 1;
end

always @ (notify_Tcph) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun positive pulse width time Tcph=%0dns \n", Tcph);
  viol_Tcph = 1;
end

always @ (notify_Tcpl) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun negative pulse width time Tcpl=%0dns \n", Tcpl);
  viol_Tcpl = 1;  
end

always @ (notify_Tcyr) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun read cycle time Tcyr=%0dns \n", Tcyr);
  viol_Tcyr = 1;
end

always @ (notify_Tcyw) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun write cycle time Tcyw=%0dns \n", Tcyw);
  viol_Tcyw = 1;  
end

always @ (notify_Ts_cel_wel) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun nCE setup time Ts_cel_wel=%0dns \n", Ts_cel_wel);
  viol_Ts_cel_wel = 1;
end

always @ (notify_Ts_oeh_wel) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun nOE setup time Ts_oeh_wel=%0dns \n", Ts_oeh_wel);
  viol_Ts_oeh_wel = 1;
end

always @ (notify_Ts_wel_cel) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun nWE setup time Ts_wel_cel=%0dns \n", Ts_wel_cel);
  viol_Ts_wel_cel = 1;
end

always @ (notify_Ts_oeh_cel) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun nOE setup time Ts_oeh_cel=%0dns \n", Ts_oeh_cel);
  viol_Ts_oeh_cel = 1;
end

always @ (notify_Th_weh_cel) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun nCE hold time Th_weh_cel=%0dns \n", Th_weh_cel);
  viol_Th_weh_cel = 1;
end

always @ (notify_Th_ceh_wel) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun nWE hold time Th_ceh_wel=%0dns \n", Th_ceh_wel);
  viol_Th_ceh_wel = 1;
end

always @ (notify_Th_oeh_weh) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun nWE hold time Th_oeh_weh=%0dns \n", Th_oeh_weh);
  viol_Th_oeh_weh = 1;
end

always @ (notify_Ts_oe_pr) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun OE_HV setup time before sector protect/unprotect pulse Ts_oe_pr=%dus \n", Ts_oe_pr);
  viol_Ts_oe_pr = 1;
end

always @ (notify_Ts_a9_pr) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun A9_HV setup time before sector protect/unprotect pulse Ts_a9_pr=%dus \n", Ts_a9_pr);
  viol_Ts_a9_pr = 1;
end

// specify section
specify
  $width (negedge nWE_IN &&& check_Twp, Twpl, , notify_Twpl); //WE# pulse width low
  $width (posedge nWE_IN &&& check_Twp, Twph, , notify_Twph); //WE# pulse width high

  $width (negedge nCE_IN &&& check_Tcp, Tcpl, , notify_Tcpl); //CE# pulse width low
  $width (posedge nCE_IN &&& check_Tcp, Tcph, , notify_Tcph); //CE# pulse width high 

	$setup (negedge nCE &&& check_Ts_cel_wel, negedge nWE_IN, Ts_cel_wel, notify_Ts_cel_wel); // use nCE to replace nCE_IN to report 0ns setup
	$setup (negedge nWE &&& check_Ts_wel_cel, negedge nCE_IN, Ts_wel_cel, notify_Ts_wel_cel); // use nWE to replace nWE_IN to report 0ns setup
	$hold  (posedge nWE_IN &&& check_Th_weh_cel, posedge nCE_IN, Th_weh_cel, notify_Th_weh_cel);
	$hold  (posedge nCE_IN &&& check_Th_ceh_wel, posedge nWE_IN, Th_ceh_wel, notify_Th_ceh_wel);

	$setup (posedge nOE_IN &&& check_Ts_oeh_wel, negedge nWE_IN, Ts_oeh_wel, notify_Ts_oeh_wel);
	$setup (posedge nOE_IN &&& check_Ts_oeh_cel, negedge nCE_IN, Ts_oeh_cel, notify_Ts_oeh_cel);
 	$hold  (posedge nWE_IN &&& check_Th_oeh_weh, negedge nOE_IN, Th_oeh_weh, notify_Th_oeh_weh); 

	$setup (A_IN &&& check_Ts_a, posedge event_d_t, Ts_a, notify_Ts_a); // Hold time 0ns shouldn't report, $hold will report,
                                                                         // so not use hold check, and describe it below
	$setup (D_IN &&& check_Ts_d, negedge event_d_t, Ts_d, notify_Ts_d);
	$hold  (negedge event_d_t &&& check_Th_d, D, Th_d, notify_Th_d);
endspecify

always @ (RSTN_PAR) begin
  if (!RSTN_PAR) begin
    viol_Ts_a = 0;
    viol_Ts_d = 0;
    viol_Th_a = 0;
    viol_Th_d = 0;

    viol_Ts_cel_wel = 0;
    viol_Ts_oeh_wel = 0;
    viol_Ts_wel_cel = 0;
    viol_Ts_oeh_cel = 0;

    viol_Th_weh_cel = 0;
    viol_Th_ceh_wel = 0;

    viol_Th_oeh_weh = 0;

    viol_Twph = 0;
    viol_Twpl = 0;
    viol_Tcph = 0;
    viol_Tcpl = 0;

    viol_Tcyr = 0;
    viol_Tcyw = 0;

    viol_Ts_oe_pr = 0;
    viol_Ts_a9_pr = 0;
  end
end

always @ (posedge event_d_t) begin
  if (($time - Tpos_event_d < Tcyw) && ($time - Tpos_event_d > 0) && ($time > 0) && check_Tcyw) begin
    notify_Tcyw = ~notify_Tcyw;
  end
  Tpos_event_d = $time;
end

always @ (A_IN) begin
  Tch_A_w = $time;
  #0; 
	if ((Tch_A_w - Tpos_event_d < Th_a) && (Tch_A_w - Tpos_event_d > 0) && (Tch_A_w > 0) && check_Th_a) begin
    notify_Th_a = ~notify_Th_a;
	end
end

always @ (negedge nCE_IN) begin
  Tneg_nCE = $time;
  #0;
  if ((Tneg_nCE - Tneg_nWE == 0) && (Tneg_nCE > 0)) begin
    ctrl_nCE = 1'b1;
    ctrl_nWE = 1'b1;
  end else if ((nWE_IN == 1'b1) && (Tneg_nCE > 0)) begin
    ctrl_nCE = 1'b0;
    ctrl_nWE = 1'b1;
  end
end

always @ (negedge nWE_IN) begin
  Tneg_nWE = $time;
  #0;
  if ((Tneg_nWE - Tneg_nCE == 0) && (Tneg_nWE > 0)) begin
    ctrl_nWE = 1'b1;
    ctrl_nCE = 1'b1;
  end else if ((nCE_IN == 1'b1) && (Tneg_nWE > 0)) begin
    ctrl_nCE = 1'b1;
    ctrl_nWE = 1'b0;
  end

  if (A9_HV_IN && OE_HV_IN && (~(nCE_IN || nWE_IN)) && (check_Ts_oe_pr || check_Ts_a9_pr)) begin
    if ( ((Tneg_nWE - Tpos_OE_HV) < (Ts_oe_pr * us_unit)) && ((Tneg_nWE - Tpos_OE_HV) > 0) && check_Ts_oe_pr ) begin
      viol_Ts_oe_pr = 1;
      notify_Ts_oe_pr = ~notify_Ts_oe_pr;
    end else begin
      viol_Ts_oe_pr = 0;      
    end 

    if ( ((Tneg_nWE - Tpos_A9_HV) < (Ts_a9_pr * us_unit)) && ((Tneg_nWE - Tpos_A9_HV) > 0) && check_Ts_a9_pr ) begin
      notify_Ts_a9_pr = ~notify_Ts_a9_pr;
      viol_Ts_a9_pr = 1;
    end else begin
      viol_Ts_a9_pr = 0;      
    end 
  end

end

always @ (posedge OE_HV_IN) Tpos_OE_HV = $time;
always @ (posedge A9_HV_IN) Tpos_A9_HV = $time;


always @ (posedge nCE_IN ) begin
  Tpos_nCE = $time;
  #0;
  if ((Tneg_nCE - Tpos_nWE >= 0) && (Tneg_nCE - Tneg_nWE > 0) && (Tpos_nCE > 0) && (nWE_IN == 1'b1)) begin
    ctrl_nCE = 1'b0;
    ctrl_nWE = 1'b0;
  end
end

always @ (posedge nWE_IN) begin
  Tpos_nWE = $time;
  #0;
  if ((Tneg_nWE - Tpos_nCE >= 0) &&  (Tneg_nWE - Tneg_nCE > 0) && (Tpos_nWE > 0) && (nCE_IN == 1'b1)) begin
    ctrl_nCE = 1'b0;
    ctrl_nWE = 1'b0;
  end
end

always @ (A_IN) begin
  if (~nCE_IN && nWE_IN) begin
    if (($time - Tch_A_r < Tcyr) && ~nOE_IN && ~nCE_IN && check_Tcyr) begin
      notify_Tcyr = ~notify_Tcyr;
    end
    Tch_A_r = $time;
  end
end

always @ (posedge nCE_IN or negedge nWE_IN) begin
  #1 Tch_A_r = 0;
end
    
//--------------------------------------------------------------

assign seq1             = (A_REG[11:0] == 15'h555) && (D_DLY == 8'hAA);
assign seq2             = (A_REG[11:0] == 15'h2AA) && (D_DLY == 8'h55);
assign seq_read_reset   = (A_REG[11:0] == 15'h555) && (D_DLY == 8'hF0);
assign seq_autoselect   = (A_REG[11:0] == 15'h555) && (D_DLY == 8'h90);
assign seq_byte_program = (A_REG[11:0] == 15'h555) && (D_DLY == 8'hA0);
assign seq_erase        = (A_REG[11:0] == 15'h555) && (D_DLY == 8'h80);
assign seq_chip_erase   = (A_REG[11:0] == 15'h555) && (D_DLY == 8'h10);
assign seq_bypass       = (A_REG[11:0] == 15'h555) && (D_DLY == 8'h20);

assign SEL_PAR = ~SEL_SERIAL & ~SEL_SPI_IN;
assign RSTN_PAR = RSTN & SEL_PAR & reset_protect;

assign event_d = !nCE_GF && nOE_GF && !nWE_GF; //SIGNAL WRITE
assign event_r  = !nCE_IN && nWE_IN && !nOE_IN; //SIGNAL READ
assign #(0, Tpdz_oe_d) OE_D = event_r;

always @ (event_r or event_d) if (!RSTN && SEL_PAR && ($time > 0)) if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Behavioral model in power-on reset state. All commmands are ignored \n");

always @ (posedge Load_ER_Sector) begin: Timeout_ER_Sector // ERASE SECTOR TIMEOUT WINDOW DESCRIPTION
  #(Timeout_Window_time-1);
  ->Event_ER_Sector;
  #(1);
  Load_ER_Sector = 0;
  DQ3 = 1'b1;
end

// Toggle status bits description
always @ (negedge event_r or negedge RSTN_PAR) 
  if (!RSTN_PAR) begin
    DQ6 <= 1'b0;
	  DQ2 <= 1'b0;
  end else begin
    case (FSM_state_PAR)
      `State_byte_program: begin
        DQ6 <= ~DQ6;
        DQ2 <= 1'b0;
      end
      `State_sector_erase, `State_chip_erase, `State_page_erase: begin
        DQ6 <= ~DQ6;
        DQ2 <= ~DQ2;
      end
      `State_read_reset: begin
        DQ6 <= 1'b0;
      end
    endcase
  end

// Address bus register
always @ (posedge event_d) begin
  #(Th_a - Tglitch);
  if (viol_Ts_a || viol_Th_a ||
      viol_Ts_cel_wel || viol_Ts_oeh_wel ||
      viol_Ts_wel_cel || viol_Ts_oeh_cel ||
      viol_Twph || viol_Twpl || viol_Tcph || viol_Tcpl ||
      viol_Tcyw) A_REG <= {A_MSB{1'bx}}; 
  else A_REG <= A_DLY;
end

// Wait for first positive edge of event_d to enable FSM 
always @ (posedge event_d or negedge RSTN_PAR) 
  if (!RSTN_PAR) begin
    was_zero_event = 1'b0;
  end else was_zero_event = 1'b1; 

// Main FSM 
always @ (negedge event_d or negedge RSTN_PAR) begin
  #(Th_d - Tglitch + 0.01);
  if (!RSTN_PAR) begin
    //---------------------------------------
	  CMD_cycle_PAR <= `Null_cycle;
	  FSM_state_PAR <= `State_read_reset; 
    //---------------------------------------
	  sector_pointer <= 0;
	  for (i = 0; i < numSect; i = i + 1) sector_buf[i] <= 0;  
	  A_PROG_PAR <= {A_MSB{1'b0}};
	  D_PROG_PAR <= {D_MSB{1'b0}};
	  bypass <= 1'b0;
	end else if (was_zero_event) begin
    case (CMD_cycle_PAR)
      `Null_cycle: begin
        if (viol_Ts_a || viol_Th_a || viol_Ts_d || viol_Th_d || 
            viol_Ts_cel_wel || viol_Ts_oeh_wel ||
            viol_Ts_wel_cel || viol_Ts_oeh_cel ||
            viol_Th_weh_cel || viol_Th_ceh_wel ||
            viol_Twph || viol_Twpl || viol_Tcph || viol_Tcpl ||
            viol_Tcyw) begin // fsm state reset in case of timing violation
          //---------------------------------------
          CMD_cycle_PAR <= `Null_cycle;
          FSM_state_PAR <= FSM_state_PAR;
          //---------------------------------------
          viol_Ts_a = 0;
          viol_Th_a = 0;
          viol_Ts_d = 0;
          viol_Th_d = 0;
          viol_Ts_cel_wel = 0;
          viol_Ts_oeh_wel = 0;
          viol_Ts_wel_cel = 0;
          viol_Ts_oeh_cel = 0;
          viol_Th_weh_cel = 0;
          viol_Th_ceh_wel = 0;
          viol_Twph = 0;
          viol_Twpl = 0;
          viol_Tcph = 0;
          viol_Tcpl = 0;
          viol_Tcyw = 0;
        end else if (bypass) begin
          if (D_DLY == 8'hA0) begin
            CMD_cycle_PAR <= `Third_cycle;
            FSM_state_PAR <= `Start_up_byte_program;	
					end else if (D_DLY == 8'h90) begin
            CMD_cycle_PAR <= `First_cycle;	
          end
				end else if ((seq1 && (FSM_state_PAR != `State_sector_erase)) || (seq1 && (FSM_state_PAR == `State_sector_erase) && !DQ3)) begin
          CMD_cycle_PAR <= `First_cycle; 
        end else if (((D_DLY == 8'hF0) && (FSM_state_PAR != `State_sector_erase)) || ((D_DLY == 8'hF0) && (FSM_state_PAR==`State_sector_erase) && DQ5_erase)) begin
          CMD_cycle_PAR <= `Null_cycle;
          FSM_state_PAR <= `State_read_reset; 
          sector_pointer <= 0;
          for (i = 0; i < numSect; i = i + 1) sector_buf[i] <= 0;  
        end else begin 
          CMD_cycle_PAR <= `Null_cycle;
          FSM_state_PAR <= FSM_state_PAR;
        end
      end
      `First_cycle: begin 
        if (viol_Ts_a || viol_Th_a || viol_Ts_d || viol_Th_d || 
            viol_Ts_cel_wel || viol_Ts_oeh_wel ||
            viol_Ts_wel_cel || viol_Ts_oeh_cel ||
            viol_Th_weh_cel || viol_Th_ceh_wel ||
            viol_Twph || viol_Twpl || viol_Tcph || viol_Tcpl ||
            viol_Tcyw) begin // fsm state reset in case of timing violation
          //---------------------------------------
          CMD_cycle_PAR <= `Null_cycle;
          FSM_state_PAR <= FSM_state_PAR;
          //---------------------------------------
          viol_Ts_a = 0;
          viol_Th_a = 0;
          viol_Ts_d = 0;
          viol_Th_d = 0;
          viol_Ts_cel_wel = 0;
          viol_Ts_oeh_wel = 0;
          viol_Ts_wel_cel = 0;
          viol_Ts_oeh_cel = 0;
          viol_Th_weh_cel = 0;
          viol_Th_ceh_wel = 0;
          viol_Twph = 0;
          viol_Twpl = 0;
          viol_Tcph = 0;
          viol_Tcpl = 0;
          viol_Tcyw = 0;
        end else if (bypass && !DQ5_write) begin
          if (D_DLY == 8'h00) begin
            bypass <= 1'b0;
          end
          CMD_cycle_PAR <= `Null_cycle;
          FSM_state_PAR <= FSM_state_PAR;  
        end else if (seq2) begin
          CMD_cycle_PAR <= `Second_cycle; 
        end else begin
          CMD_cycle_PAR <= `Null_cycle;
          FSM_state_PAR <= FSM_state_PAR;  
        end
      end
      `Second_cycle: begin
        if (viol_Ts_a || viol_Th_a || viol_Ts_d || viol_Th_d || 
            viol_Ts_cel_wel || viol_Ts_oeh_wel ||
            viol_Ts_wel_cel || viol_Ts_oeh_cel ||
            viol_Th_weh_cel || viol_Th_ceh_wel ||
            viol_Twph || viol_Twpl || viol_Tcph || viol_Tcpl ||
            viol_Tcyw) begin // fsm state reset in case of timing violation
          //---------------------------------------
          CMD_cycle_PAR <= `Null_cycle;
          FSM_state_PAR <= FSM_state_PAR;
          //---------------------------------------
          viol_Ts_a = 0;
          viol_Th_a = 0;
          viol_Ts_d = 0;
          viol_Th_d = 0;
          viol_Ts_cel_wel = 0;
          viol_Ts_oeh_wel = 0;
          viol_Ts_wel_cel = 0;
          viol_Ts_oeh_cel = 0;
          viol_Th_weh_cel = 0;
          viol_Th_ceh_wel = 0;
          viol_Twph = 0;
          viol_Twpl = 0;
          viol_Tcph = 0;
          viol_Tcpl = 0;
          viol_Tcyw = 0;
        end else if ((FSM_state_PAR == `State_sector_erase) && !seq_erase) begin
          CMD_cycle_PAR <= `Null_cycle;
          FSM_state_PAR <= `State_read_reset; 
          sector_pointer <= 0;
          for (i = 0; i < numSect; i = i + 1) sector_buf[i] <= 0;  
        end else if ((FSM_state_PAR == `State_sector_erase) && seq_erase) begin
          CMD_cycle_PAR <= `Third_cycle;
        end else if (seq_autoselect && !DQ5_erase && !DQ5_write) begin  //START-UP AUTOSELECT
          CMD_cycle_PAR <= `Null_cycle;
          FSM_state_PAR <= `State_autoselect;
        end else if (seq_byte_program && !DQ5_erase && !DQ5_write) begin  //START-UP BYTE PROGRAMM
          CMD_cycle_PAR <= `Third_cycle;
          FSM_state_PAR <= `Start_up_byte_program;
        end else if (seq_erase && !DQ5_erase && !DQ5_write) begin //START-UP ERASE
          CMD_cycle_PAR <= `Third_cycle;
          FSM_state_PAR <= `State_erase;
        end else if (seq_read_reset) begin // START-UP RESET/READ
          CMD_cycle_PAR <= `Null_cycle;
          FSM_state_PAR <= `State_read_reset;
        end else if (seq_bypass) begin
          CMD_cycle_PAR <= `Null_cycle;
          FSM_state_PAR <= FSM_state_PAR; 
          bypass <= 1'b1;
        end else begin
          CMD_cycle_PAR <= `Null_cycle;
          FSM_state_PAR <= FSM_state_PAR; 
        end
      end
      `Third_cycle: begin
        if (viol_Ts_a || viol_Th_a || viol_Ts_d || viol_Th_d || 
            viol_Ts_cel_wel || viol_Ts_oeh_wel ||
            viol_Ts_wel_cel || viol_Ts_oeh_cel ||
            viol_Th_weh_cel || viol_Th_ceh_wel ||
            viol_Twph || viol_Twpl || viol_Tcph || viol_Tcpl ||
            viol_Tcyw) begin // fsm state reset in case of timing violation
          //---------------------------------------
          CMD_cycle_PAR <= `Null_cycle;
          FSM_state_PAR <= `State_read_reset;
          //---------------------------------------
          sector_pointer <= 0;
          for (i = 0; i < numSect; i = i + 1) sector_buf[i] <= 0; 
          viol_Ts_a = 0;
          viol_Th_a = 0;
          viol_Ts_d = 0;
          viol_Th_d = 0;
          viol_Ts_cel_wel = 0;
          viol_Ts_oeh_wel = 0;
          viol_Ts_wel_cel = 0;
          viol_Ts_oeh_cel = 0;
          viol_Th_weh_cel = 0;
          viol_Th_ceh_wel = 0;
          viol_Twph = 0;
          viol_Twpl = 0;
          viol_Tcph = 0;
          viol_Tcpl = 0;
          viol_Tcyw = 0;
        end else if ((FSM_state_PAR==`State_sector_erase) && !seq1) begin
          CMD_cycle_PAR <= `Null_cycle;
          FSM_state_PAR <= `State_read_reset;
          sector_pointer <= 0;
          for (i = 0; i < numSect; i = i + 1) sector_buf[i] <= 0;  
        end else if (seq1 && ((FSM_state_PAR == `State_erase) || (FSM_state_PAR == `State_sector_erase))) begin
          CMD_cycle_PAR<=`Fourth_cycle; 	//FOURTH CYCLE ERASE
        end else if (FSM_state_PAR == `Start_up_byte_program) begin //RUN WRITE BYTE PROGRAM
          FSM_state_PAR <= `State_byte_program;
          CMD_cycle_PAR <= `Sixth_cycle;
          A_PROG_PAR <= A_REG;
          D_PROG_PAR <= D_DLY;
          #1;
          ->Event_WR;
        end else begin
          CMD_cycle_PAR <= `Null_cycle;
          FSM_state_PAR <= `State_read_reset;
          sector_pointer <= 0;
          for (i = 0; i < numSect; i = i + 1) sector_buf[i] <= 0;  
        end	
      end
      `Fourth_cycle: begin
        if (viol_Ts_a || viol_Th_a || viol_Ts_d || viol_Th_d || 
            viol_Ts_cel_wel || viol_Ts_oeh_wel ||
            viol_Ts_wel_cel || viol_Ts_oeh_cel ||
            viol_Th_weh_cel || viol_Th_ceh_wel ||
            viol_Twph || viol_Twpl || viol_Tcph || viol_Tcpl ||
            viol_Tcyw) begin // fsm state reset in case of timing violation
          //---------------------------------------
          CMD_cycle_PAR <= `Null_cycle;
          FSM_state_PAR <= `State_read_reset;
          //---------------------------------------
          sector_pointer <= 0;
          for (i = 0; i < numSect; i = i + 1) sector_buf[i] <= 0; 
          viol_Ts_a = 0;
          viol_Th_a = 0;
          viol_Ts_d = 0;
          viol_Th_d = 0;
          viol_Ts_cel_wel = 0;
          viol_Ts_oeh_wel = 0;
          viol_Ts_wel_cel = 0;
          viol_Ts_oeh_cel = 0;
          viol_Th_weh_cel = 0;
          viol_Th_ceh_wel = 0;
          viol_Twph = 0;
          viol_Twpl = 0;
          viol_Tcph = 0;
          viol_Tcpl = 0;
          viol_Tcyw = 0;
        end else if (seq2) begin
          CMD_cycle_PAR <= `Fifth_cycle;  //FIFTH CYCLE ERASE
        end else begin
          CMD_cycle_PAR <= `Null_cycle;
          FSM_state_PAR <= `State_read_reset;
          sector_pointer <= 0;
          for (i = 0; i < numSect; i = i + 1) sector_buf[i] <= 0;  
        end		  
      end
      `Fifth_cycle: begin
        if (viol_Ts_a || viol_Th_a || viol_Ts_d || viol_Th_d || 
            viol_Ts_cel_wel || viol_Ts_oeh_wel ||
            viol_Ts_wel_cel || viol_Ts_oeh_cel ||
            viol_Th_weh_cel || viol_Th_ceh_wel ||
            viol_Twph || viol_Twpl || viol_Tcph || viol_Tcpl ||
            viol_Tcyw) begin // fsm state reset in case of timing violation
          //---------------------------------------
          CMD_cycle_PAR <= `Null_cycle;
          FSM_state_PAR <= `State_read_reset;
          //---------------------------------------
          sector_pointer <= 0;
          for (i = 0; i < numSect; i = i + 1) sector_buf[i] <= 0; 
          viol_Ts_a = 0;
          viol_Th_a = 0;
          viol_Ts_d = 0;
          viol_Th_d = 0;
          viol_Ts_cel_wel = 0;
          viol_Ts_oeh_wel = 0;
          viol_Ts_wel_cel = 0;
          viol_Ts_oeh_cel = 0;
          viol_Th_weh_cel = 0;
          viol_Th_ceh_wel = 0;
          viol_Twph = 0;
          viol_Twpl = 0;
          viol_Tcph = 0;
          viol_Tcpl = 0;
          viol_Tcyw = 0;
        end else if (seq_chip_erase && (FSM_state_PAR != `State_sector_erase)) begin //RUN CHIP ERASE 
          CMD_cycle_PAR <= `Sixth_cycle;
          FSM_state_PAR <= `State_chip_erase;
          #1;
          ->Event_ER_Chip;
        end else if ((D_DLY == 8'h50) && (FSM_state_PAR != `State_sector_erase)) begin //RUN PAGE ERASE 
          CMD_cycle_PAR <= `Sixth_cycle;
          FSM_state_PAR <= `State_page_erase; 
          A_PROG_PAR <= A_REG;
          #1;
          ->Event_ER_Page;
        end else if ((D_DLY == 8'h30) && !DQ3) begin  //RUN SECTOR ERASE 
          CMD_cycle_PAR <= `Null_cycle;
          FSM_state_PAR <= `State_sector_erase;
          if (sector_pointer < numSect) begin
            sector_buf[sector_pointer] <= A_REG[A_MSB-1:A_SEC_MSB];
            sector_pointer <= sector_pointer + 1; // start time-out window 80 us
            Load_ER_Sector = 1'b0;
            disable Timeout_ER_Sector;
            #1;
            Load_ER_Sector = 1'b1;
          end
        end else if ((D_DLY == 8'h30) && DQ3) begin //CONTINUE SECTOR ERASE 
          CMD_cycle_PAR <= `Null_cycle;
          FSM_state_PAR <= FSM_state_PAR;
        end else begin
          CMD_cycle_PAR <= `Null_cycle;
          FSM_state_PAR <= `State_read_reset;
          sector_pointer <= 0;
          for (i = 0; i < numSect; i  = i + 1) sector_buf[i] <= 0;  
        end	
      end
      `Sixth_cycle: begin
        if (viol_Ts_a || viol_Th_a || viol_Ts_d || viol_Th_d || 
            viol_Ts_cel_wel || viol_Ts_oeh_wel ||
            viol_Ts_wel_cel || viol_Ts_oeh_cel ||
            viol_Th_weh_cel || viol_Th_ceh_wel ||
            viol_Twph || viol_Twpl || viol_Tcph || viol_Tcpl ||
            viol_Tcyw) begin // fsm state reset in case of timing violation
          //---------------------------------------
          CMD_cycle_PAR <= CMD_cycle_PAR;
          FSM_state_PAR <= FSM_state_PAR; 
          //---------------------------------------
          viol_Ts_a = 0;
          viol_Th_a = 0;
          viol_Ts_d = 0;
          viol_Th_d = 0;
          viol_Ts_cel_wel = 0;
          viol_Ts_oeh_wel = 0;
          viol_Ts_wel_cel = 0;
          viol_Ts_oeh_cel = 0;
          viol_Th_weh_cel = 0;
          viol_Th_ceh_wel = 0;
          viol_Twph = 0;
          viol_Twpl = 0;
          viol_Tcph = 0;
          viol_Tcpl = 0;
          viol_Tcyw = 0;
        end else if ((DQ5_write || DQ5_erase) && (D_DLY == 8'hF0)) begin //RESET PROGRAM OR ERASE MODE
          CMD_cycle_PAR <= `Null_cycle;
          FSM_state_PAR <= `State_read_reset;
          sector_pointer <= 0;
          for (i = 0; i < numSect; i = i + 1) sector_buf[i] <= 0; 
          DQ5_write <= 1'b0; 
          DQ5_erase <= 1'b0; 
        end else if ((DQ5_write || DQ5_erase) && seq1) begin  //RESET PROGRAM OR ERASE MODE
          CMD_cycle_PAR <= `First_cycle;
          FSM_state_PAR <= FSM_state_PAR;
          DQ5_write <= 1'b0; 
          DQ5_erase <= 1'b0; 
        end else begin
          CMD_cycle_PAR <= CMD_cycle_PAR;
          FSM_state_PAR <= FSM_state_PAR;  
        end
      end
      default: begin
      end
    endcase
  end
end
						
//==============================================================================================================================================
//==============================================================================================================================================
// SERIAL I/F
//==============================================================================================================================================
//==============================================================================================================================================
`define Full_cycle_mode   				2'b00
`define Command_cycle_mode 			  2'b01
`define Read_cycle_mode   			  2'b10 
`define Serial_operation_mode   	2'b11 

/*----------------------------------------------------------------------*/
/* Specify the timing                                                   */
/*----------------------------------------------------------------------*/
reg notify_T_tck_min,    viol_T_tck_min;
reg notify_T_tck_er_max, viol_T_tck_er_max;
reg notify_T_tck_er_min, viol_T_tck_er_min;
reg notify_T_tck_wr_max, viol_T_tck_wr_max;
reg notify_T_tck_wr_min, viol_T_tck_wr_min;

reg notify_Ts_tck_tdi, viol_Ts_tck_tdi;
reg notify_Th_tdi_tck, viol_Th_tdi_tck;

wire check_T_tck_min = RSTN && SEL_SERIAL_work;
wire check_T_tck_er = RSTN && SEL_SERIAL_work && Process_ER_Sector_tst;
wire check_T_tck_wr = RSTN && SEL_SERIAL_work && Process_WR_tst;

wire check_Ts_tck_tdi = SEL_SERIAL_work;
wire check_Th_tdi_tck = SEL_SERIAL_work;

time Tpos_tck_last;
time Tpos_tck_current;
time T_tck;

initial begin
  notify_T_tck_min = 0;
  notify_T_tck_er_max = 0;
  notify_T_tck_er_min = 0;
  notify_T_tck_wr_max = 0;
  notify_T_tck_wr_min = 0;

  notify_Ts_tck_tdi = 0;
  notify_Th_tdi_tck = 0;

  viol_T_tck_min = 0;
  viol_T_tck_er_max = 0;
  viol_T_tck_er_min = 0;
  viol_T_tck_wr_max = 0;
  viol_T_tck_wr_min = 0;

  viol_Ts_tck_tdi = 0;
  viol_Th_tdi_tck = 0;

  Tpos_tck_last = 0;
  Tpos_tck_current = 0;
  T_tck = 0;
end

// Notify section
always @ (notify_Ts_tck_tdi) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun TDI setup time Ts_tck_tdi=%0dns \n", Ts_tck_tdi);
  viol_Ts_tck_tdi = 1'b1;
end

always @ (notify_Th_tdi_tck) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun TDI hold time Th_tdi_tck=%0dns \n", Th_tdi_tck);
  viol_Th_tdi_tck = 1'b1;
end

always @ (notify_T_tck_min) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Maximum TCK frequency is 50MHz \n");
end

always @ (notify_T_tck_er_max) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Required maximum TCK period for erase operation must be Tc(ER)<%0dns \n", T_tck_er_max);
end

always @ (notify_T_tck_er_min) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Required minimum TCK period for erase operation must be Tc(ER)>%0dns \n", T_tck_er_min);
end

always @ (notify_T_tck_wr_max) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Required maximum TCK period for program operation must be Tc(WR)<%0dns \n", T_tck_wr_max);
end

always @ (notify_T_tck_wr_min) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Required minimum TCK period for program operation must be Tc(WR)>%0dns \n", T_tck_wr_min);
end

// Specify section
always @ (posedge TCK_IN) begin: measure_TCK_period
  Tpos_tck_last = Tpos_tck_current;
  Tpos_tck_current = $time;
  T_tck = Tpos_tck_current - Tpos_tck_last;
  #0;
  if ( (T_tck < T_tck_min) && (T_tck > 0) && check_T_tck_min) begin
    viol_T_tck_min = 1;
    notify_T_tck_min = ~notify_T_tck_min;
  end else begin
    viol_T_tck_min = 0;
  end

  if ( (T_tck > T_tck_er_max) && (T_tck > 0) && check_T_tck_er) begin
    viol_T_tck_er_max = 1;
    notify_T_tck_er_max = ~notify_T_tck_er_max;
  end else begin
    viol_T_tck_er_max = 0;
  end

  if ( (T_tck < T_tck_er_min) && (T_tck > 0) && check_T_tck_er) begin
    viol_T_tck_er_min = 1;
    notify_T_tck_er_min = ~notify_T_tck_er_min;
  end else begin
    viol_T_tck_er_min = 0;
  end  

  if ( (T_tck > T_tck_wr_max) && (T_tck > 0) && check_T_tck_wr) begin
    viol_T_tck_wr_max = 1;
    notify_T_tck_wr_max = ~notify_T_tck_wr_max;
  end else begin
    viol_T_tck_wr_max = 0;
  end

  if ( (T_tck < T_tck_wr_min) && (T_tck > 0) && check_T_tck_wr) begin
    viol_T_tck_wr_min = 1;
    notify_T_tck_wr_min = ~notify_T_tck_wr_min;
  end else begin
    viol_T_tck_wr_min = 0;
  end
end

specify
  $setup (TDI_IN &&& check_Ts_tck_tdi, posedge TCK_IN, Ts_tck_tdi, notify_Ts_tck_tdi);
  $hold (posedge TCK_IN &&& check_Th_tdi_tck, TDI_IN, Th_tdi_tck, notify_Th_tdi_tck);
endspecify
// ------------------------------------------------------------------------

assign MRST_work = RSTN && ~SEL_SPI_IN && MRST_IN;
assign SEL_SERIAL_work = SEL_SERIAL;
assign TDI_IN_work = TDI_IN;

assign MRST_STROBE = MRST_work && ~STROBE_IN; 

always @ (posedge TCK_IN) 
  if (!RSTN && ~SEL_SPI_IN && MRST_IN) 
    if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Behavioral model in power-on reset state. All commmands are ignored \n");

always @ (posedge TCK_IN or negedge MRST_work)
  if (!MRST_work) SEL_SERIAL <= 1'b0;
  else if (STROBE_IN) begin
    if (!SEL_SERIAL) if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Successfully entered to the serial mode \n");
    SEL_SERIAL <= 1'b1;
  end

always @ (posedge TCK or negedge MRST_work)
  if (!MRST_work) begin
    A_SERIAL <= {A_MSB{1'b0}};
	  DIN_SERIAL <= {D_MSB{1'b0}};
	  State_current <= 2'b00;
	  State_previous <= 2'b00;
	  count_bit <= 6'h0;
	  command <= 11'h0;
	  address <= 17'h0;
	  data <= 16'h0;
	  rep_eat <= 0;	 
    read_flag <= 1'b0;
	  seq <= 8'h0;
    RDY_BSY_SERIAL <= 1'b0;
	end else if (STROBE_IN) begin 
	  count_bit <= 6'h0;
    rep_eat <= 1'b0;
    data <= 16'h0;
  end else if (SEL_SERIAL) begin
    if (viol_Ts_tck_tdi || viol_Th_tdi_tck || viol_T_tck_min) begin
      A_SERIAL <= {A_MSB{1'b0}};
	    State_current <= 2'b00;
	    State_previous <= 2'b00;   
	    count_bit <= 6'h0;   
	    command <= 11'h0;
	    address <= 17'h0;
	    data <= 16'h0; 
      viol_Ts_tck_tdi <= 1'b0;
      viol_Th_tdi_tck <= 1'b0;
    end else begin
    case (count_bit)
      6'd00: begin 
        State_current[1] <= TDI_IN_work;
        count_bit <= count_bit + 1;
      end
      6'd01: begin 
        State_current[0] <= TDI_IN_work;
        count_bit <= count_bit + 1;
        if (({State_current[1],TDI_IN_work}==2'b00)||({State_current[1],TDI_IN_work}==2'b10)||({State_current[1],TDI_IN_work}==2'b01)) State_previous<={State_current[1],TDI_IN_work};
      end
      default: begin
        if ((count_bit>=6'd02)&&(count_bit<=6'd12)&&(State_current!=`Serial_operation_mode)) begin
          command <= {TDI_IN_work,command[10:1]};
          count_bit <= count_bit + 1;
        end else if ((count_bit>=6'd13)&&(count_bit<=6'd29)&&((State_current==`Full_cycle_mode)||(State_current==`Read_cycle_mode))) begin
          address <= {TDI_IN_work, address[16:1]};
          if ((count_bit==6'd13)&&command[1]) count_bit <= count_bit + 2;
          else count_bit <= count_bit + 1;
          if ((count_bit==6'd29) && (State_current==`Full_cycle_mode)) begin	
            A_SERIAL[A_SEC_MSB-2:1] <= {TDI_IN_work, address[16:2]};
            if (!command[1]) A_SERIAL[0] <= address[1];
            else A_SERIAL[0] <= 1'b0;	
          end
        end	else if ((count_bit>=6'd13)&&(count_bit<=6'd24)&&(State_current==`Command_cycle_mode)) begin
          address <= {5'b00000, TDI_IN_work, address[11:1]};
          count_bit <= count_bit + 1;
          if (count_bit==6'd24) A_SERIAL[A_SEC_MSB-1:0] <= {6'b000000, TDI_IN_work, address[11:1]};
        end else if ((count_bit>=6'd25)&&(count_bit<=6'd32)&&(State_current==`Command_cycle_mode)) begin
          data <= {TDI_IN_work, data[7:1]};
          count_bit <= count_bit + 1;
        end else if ((count_bit>=6'd30)&&(count_bit<=6'd37)&&(State_current==`Full_cycle_mode)) begin
          if (command[1]==1'b0) data <= {8'h00, TDI_IN_work, data[7:1]};
          else data <= {TDI_IN_work, data[15:1]};
          if (command[1]&&rep_eat) count_bit <= count_bit + 1;
          if (command[1]==1'b0) count_bit <= count_bit + 1;
          rep_eat <= ~rep_eat;
        end else if ((count_bit>=6'd02)&&(count_bit<=6'd09)&&(State_previous==`Full_cycle_mode)&&(State_current==`Serial_operation_mode)) begin
          if (command[1]==1'b0) data <= {8'h00, TDI_IN_work, data[7:1]};
          else data <= {TDI_IN_work, data[15:1]};
          if (command[1]&&rep_eat) count_bit <= count_bit + 1;
          if (command[1]==1'b0) count_bit <= count_bit + 1;
          rep_eat <= ~rep_eat;
        end else if (((count_bit==6'd38)&&(State_current==`Full_cycle_mode))||((count_bit==6'd33)&&(State_current==`Command_cycle_mode))) begin  
          casez({seq,A_SERIAL[11:0],data[7:0]})
            {8'h1,12'hAAA,8'h55},  				  
            {8'h2,12'h555,8'hA0},  				   
            {8'h10,12'h555,8'hAA},  			 
            {8'h20,12'hAAA,8'h55},	
            {8'h4,12'b????????????,8'b????????},
            {8'h40,11'h555,1'b?,8'h10}, 
            {8'h40,12'h555,8'h10}, 
            {8'h40,12'h000,8'h50}: seq <= seq<<1;
            {8'h0,12'h555,8'hAA}: seq <= 8'h1;
            {8'h2,12'h555,8'h80}: seq <= seq<<3;  
            default: seq <= 8'h0;
          endcase	
          count_bit <= count_bit + 1;
        end else if (((count_bit==6'd39)&&(State_current==`Full_cycle_mode))||((count_bit==6'd34)&&(State_current==`Command_cycle_mode))) begin
          if (seq[7] | seq[3])	begin	 
            DIN_SERIAL <= data[7:0];	
            A_SERIAL[A_SEC_MSB-1] <= command[7]; 
            A_SERIAL[A_MSB-1:A_SEC_MSB] <= command[10:8];
            seq <= 8'h0;
            read_flag <= 1'b0;
            if (command[0]) if (CONSOLE_LOG && EXTRA_LOG) $display("\nTime:%0dns -> ", $time, "Command bit VREAD must be 0 state \n");
            if (!command[5]) if (CONSOLE_LOG && EXTRA_LOG) $display("\nTime:%0dns -> ", $time, "Command bit NVRb is not supported by model \n");
						/////////////////////////////////
            if (seq[7]) ->Event_ER_Sector_tst;
            if (seq[3]) ->Event_WR_tst;
            RDY_BSY_SERIAL = 1'b1;
						/////////////////////////////////
          end if (seq[3]) count_bit <= count_bit + 2;
          else count_bit <= count_bit + 1;
        end else if (((count_bit==6'd40)&&(State_current==`Full_cycle_mode))||((count_bit==6'd35)&&(State_current==`Command_cycle_mode))) begin
          count_bit <= count_bit;
          seq[3] <= 1'b0;
          seq[7] <= 1'b0;
          /////////////////////////////////
          if (seq[3]) ->Event_WR_tst;
          if (seq[3]) RDY_BSY_SERIAL = 1'b1;
          /////////////////////////////////
        end	else if (((count_bit==6'd41)&&(State_current==`Full_cycle_mode))||((count_bit==6'd35)&&(State_current==`Command_cycle_mode))) begin
          count_bit <= count_bit;
          if (command[1] && !RDY_BSY_SERIAL && (State_current==`Full_cycle_mode)) begin
            seq[3] <= 1'b1;
            DIN_SERIAL <= data[15:8];  
            A_SERIAL[0] <= 1'b1; 
            count_bit <= 6'd40;
          end else begin
            seq <= 8'h0;
            seq[3] <= 1'b0;
            seq[7] <= 1'b0;
          end
        end else if ((count_bit==6'd10)&&(State_previous==`Full_cycle_mode)&&(State_current==`Serial_operation_mode)) begin
          count_bit <= count_bit + 1;	
          seq[3] <= 1'b1;
          if (command[1]==1'b0) begin
            address[16:0] <= address[16:0] + 1;
            A_SERIAL[A_SEC_MSB-2:0] <= address[16:0] + 1;
          end else begin
            address[16:0] <= address[16:0] + 2;
            A_SERIAL[A_SEC_MSB-2:1] <= address[16:1] + 1;
            A_SERIAL[0] <= 1'b0;
          end
          DIN_SERIAL <= data[7:0];
        end else if ((count_bit==6'd11)&&(State_previous==`Full_cycle_mode)&&(State_current==`Serial_operation_mode)) begin
          count_bit <= count_bit + 1;
          seq[3] <= 1'b0;
          /////////////////////////////////
          if (seq[3]) ->Event_WR_tst;
          if (seq[3]) RDY_BSY_SERIAL = 1'b1;
          /////////////////////////////////
        end else if ((count_bit==6'd12)&&(State_previous==`Full_cycle_mode)&&(State_current==`Serial_operation_mode)) begin
          count_bit <= count_bit;
          if (command[1] & !RDY_BSY_SERIAL) begin
            seq[3] <= 1'b1;
            DIN_SERIAL <= data[15:8];  
            A_SERIAL[0] <= 1'b1; 
            count_bit <= count_bit + 1;
          end	
        end else if ((count_bit==6'd13)&&(State_previous==`Full_cycle_mode)&&(State_current==`Serial_operation_mode)) begin
          count_bit <= count_bit;
          seq[3] <= 1'b0;
          /////////////////////////////////
          if (seq[3]) ->Event_WR_tst;
          if (seq[3]) RDY_BSY_SERIAL = 1'b1;
          /////////////////////////////////
        end else if ((count_bit==6'd30)&&(State_current==`Read_cycle_mode))begin
          if (!RDY_BSY_SERIAL) begin
            count_bit <= count_bit + 1;
            A_SERIAL[A_SEC_MSB-1] <= command[7]; 
            A_SERIAL[A_SEC_MSB-2:1] <= address[16:1];
            read_flag <= !command[4];
            if (!command[1]) A_SERIAL[0] <= address[0];
            else A_SERIAL[0] <= 1'b0;	
            A_SERIAL[A_MSB-1:A_SEC_MSB] <= command[10:8];        
            seq <= 8'h0;
          end else begin 	
            count_bit <= count_bit;
          end 
        end  else if ((count_bit==6'd31)&&(State_current==`Read_cycle_mode)) begin
          if (!command[1]) begin
            address[16:0] <= address[16:0] + 1;
            A_SERIAL[A_SEC_MSB-2:0] <= address[16:0] + 1;
          end else begin
            address[16:0] <= {address[16:1],1'b0} + 2;
            A_SERIAL[A_SEC_MSB-2:0] <= {address[16:1],1'b1};
          end
          count_bit <= count_bit + 1;
          data <= {9'h000,ARRAY[A_SERIAL][7:1]};
        end else if ((count_bit==6'd32)&&(State_current==`Read_cycle_mode)) begin
          if (A_SERIAL[0] && command[1]) data <= {2'b00,ARRAY[A_SERIAL][7:0],data[6:1]};
          else data <= {1'b0,data[15:1]};
          A_SERIAL[A_SEC_MSB-2:0] <= address[16:0];
        end else if ((count_bit==6'd02)&&(State_previous==`Read_cycle_mode)&&(State_current==`Serial_operation_mode)) begin
          count_bit <= count_bit + 1;	 
          seq <= 8'h0;
          data <= {9'h000,ARRAY[A_SERIAL][7:1]};
          if (command[1]==1'b0) begin
            address[16:0] <= address[16:0] + 1;
            A_SERIAL[A_SEC_MSB-2:0] <= address[16:0] + 1;
          end else begin
            address[16:0] <= {address[16:1],1'b0} + 2;
            A_SERIAL[A_SEC_MSB-2:0] <= {address[16:1],1'b1};
          end
        end else if ((count_bit==6'd03)&&(State_previous==`Read_cycle_mode)&&(State_current==`Serial_operation_mode)) begin
          if (A_SERIAL[0] && command[1]) data <= {2'b00, ARRAY[A_SERIAL][7:0],data[6:1]}; 
          else data <= {1'b0,data[15:1]};
          A_SERIAL[A_SEC_MSB-2:0] <= address[16:0];
        end
      end
    endcase
    end
  end

always @(negedge TCK_IN or negedge MRST_STROBE)
  if (MRST_STROBE==1'b0) begin
    #(Tpdz_sck_d);
    TDI_OUT <= 1'b1;
    OE_TDO <= 1'b0;
	end else if (SEL_SERIAL) begin
    if (((count_bit>=6'd38)&&(State_current==`Full_cycle_mode))||((count_bit>=6'd10)&&(State_previous==`Full_cycle_mode)&&(State_current==`Serial_operation_mode))
				||((count_bit==6'd30)&&(State_current==`Read_cycle_mode))) begin
      TDI_OUT <= 1'bx;
      #(Tpzd_tck_d);
      OE_TDO <= 1'b1;
      #(Ta_tck - Tpzd_tck_d);
      if (viol_T_tck_min) TDI_OUT <= 1'bx;
      else TDI_OUT<=((seq[3] || (count_bit==6'd12) || ((count_bit==6'd41)&&(State_current==`Full_cycle_mode))) && command[1] & !RDY_BSY_SERIAL)?1'b0:!RDY_BSY_SERIAL;
    end else if (((count_bit==6'd31)&&(State_current==`Read_cycle_mode)&&read_flag)||((count_bit==6'd02)&&(State_previous==`Read_cycle_mode)&&(State_current==`Serial_operation_mode)&&read_flag)) begin
      TDI_OUT <= 1'bx;
      #(Tpzd_tck_d);
      OE_TDO <= 1'b1;
      #(Ta_tck - Tpzd_tck_d);
      if (viol_T_tck_min) begin
        if (CONSOLE_LOG && EXTRA_LOG) $display("\nTime:%0dns -> ", $time, "Unable to read data from address: %h \n", A_SERIAL);
        TDI_OUT <= 1'bx;
      end else begin
        if (CONSOLE_LOG && EXTRA_LOG) $display("\nTime:%0dns -> ", $time, "Data: %h read from address: %h \n", ARRAY[A_SERIAL], A_SERIAL);
        TDI_OUT <= ARRAY[A_SERIAL][0];
      end
    end else if (((count_bit==6'd32)&&(State_current==`Read_cycle_mode)&&read_flag)||((count_bit==6'd03)&&(State_previous==`Read_cycle_mode)&&(State_current==`Serial_operation_mode)&&read_flag)) begin
      TDI_OUT <= 1'bx;
      #(Tpzd_tck_d);
      OE_TDO <= 1'b1;
      #(Ta_tck - Tpzd_tck_d);
      if (viol_T_tck_min) TDI_OUT <= 1'bx;
      else TDI_OUT <= data[0];
    end
  end

//==============================================================================================================================================
//==============================================================================================================================================
// SPI I/F
//==============================================================================================================================================
//==============================================================================================================================================
`define OP_RD1        8'h0B
`define OP_RD2        8'h03
`define OP_WR         8'h02
`define OP_ER1        8'hD8
`define OP_ER2        8'h60
`define OP_WEL        8'h06
`define OP_WDL        8'h04
`define OP_Protect    8'h36
`define OP_Unprotect  8'h39
`define OP_RDStatus   8'h05
`define OP_WRStatus   8'h01
`define OP_RDPrReg    8'h3C
`define OP_RDID       8'h9F
`define OP_Reset      8'hF0

/*----------------------------------------------------------------------*/
/* Specify the timing                                                   */
/*----------------------------------------------------------------------*/
time Tpos_ce_spi;
time Tneg_ce_spi;

time Tneg_sck_last;
time Tneg_sck_current;
time T_sck;

reg notify_Twh_sck, viol_Twh_sck;
reg notify_Twl_sck, viol_Twl_sck;

reg notify_Ts_sck_si, viol_Ts_sck_si;
reg notify_Th_si_sck, viol_Th_si_sck;

reg notify_Ts1_ce, viol_Ts1_ce;
reg notify_Ts2_ce;
reg notify_Th1_ce, viol_Th1_ce;
reg notify_Th2_ce;

reg notify_Twh1_ce;
reg notify_Twh2_ce, viol_Twh2_ce;

reg notify_T_sck_rd1, viol_T_sck_rd1;
reg notify_T_sck_rd2, viol_T_sck_rd2;
reg notify_T_sck_min, viol_T_sck_min;

wire check_T_sck_rd1 = RSTN && incycle && SEL_SPI_work && (bit_counter>=6'h08) && 
                        ((opcode==`OP_RD1) || (opcode==`OP_RDStatus) || 
                        (opcode==`OP_RDPrReg) || (opcode==`OP_RDID));

wire check_T_sck_rd2 = RSTN && incycle && SEL_SPI_work && (bit_counter>=6'h08) && 
                        (opcode==`OP_RD2);

wire check_T_sck_min = RSTN && incycle && SEL_SPI_work;

wire check_Twh_sck = RSTN && incycle && SEL_SPI_work;
wire check_Twl_sck = RSTN && incycle && SEL_SPI_work;

wire check_Ts_sck_si = RSTN && SEL_SPI_work;
wire check_Th_si_sck = RSTN && SEL_SPI_work;

wire check_Ts1_ce = RSTN && SEL_SPI_work;
wire check_Ts2_ce = RSTN && SEL_SPI_work;
wire check_Th1_ce = RSTN && SEL_SPI_work;
wire check_Th2_ce = RSTN && SEL_SPI_work;

wire check_Twh1_ce = RSTN && SEL_SPI_work && execute_opcode;
wire check_Twh2_ce = RSTN && SEL_SPI_work;

initial begin
  Tpos_ce_spi = 0;
  Tneg_ce_spi = 0;

  Tneg_sck_last = 0;
  Tneg_sck_current = 0;
  T_sck = 0;

  notify_Twh_sck = 0;
  notify_Twl_sck = 0;

  notify_Ts_sck_si = 0;
  notify_Th_si_sck = 0;

  notify_Ts1_ce = 0;
  notify_Ts2_ce = 0;
  notify_Th1_ce = 0;
  notify_Th2_ce = 0;

  notify_Twh1_ce = 0;
  notify_Twh2_ce = 0;

  notify_T_sck_rd1 = 0;
  notify_T_sck_rd2 = 0;
  notify_T_sck_min = 0;

  viol_T_sck_rd1 = 0;
  viol_T_sck_rd2 = 0;
  viol_T_sck_min = 0;

  viol_Twh_sck = 0;
  viol_Twl_sck = 0;

  viol_Ts_sck_si = 0;
  viol_Th_si_sck = 0;

  viol_Ts1_ce = 0;
  viol_Th1_ce = 0;

  viol_Twh2_ce = 0;
end

// Notify section
always @ (notify_Twh_sck) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun SCK positive pulse width time Twh_sck=%0dns \n", Twh_sck);
  viol_Twh_sck = 1;
end

always @ (notify_Twl_sck) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun SCK negative pulse width time Twl_sck=%0dns \n", Twl_sck);
  viol_Twl_sck = 1;
end

always @ (notify_Ts_sck_si) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun SI setup time Ts_sck_si=%0dns \n", Ts_sck_si);
  viol_Ts_sck_si = 1;
end

always @ (notify_Th_si_sck) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun SI hold time Th_si_sck=%0dns \n", Th_si_sck);
  viol_Th_si_sck = 1;
end

always @ (notify_Ts1_ce) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun nCE setup time Ts1_ce=%0dns \n", Ts1_ce);
  viol_Ts1_ce = 1;
end

always @ (notify_Th1_ce) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun nCE hold time Th1_ce=%0dns \n", Th1_ce);
  viol_Th1_ce = 1;
end

always @ (notify_Ts2_ce) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun nCE setup time Ts2_ce=%0dns \n", Ts2_ce);
end

always @ (notify_Th2_ce) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun nCE hold time Th2_ce=%0dns \n", Th2_ce);
end

always @ (notify_Twh1_ce) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun nCE positive pulse width time for write operation Twh1_ce=%0dns \n", Twh1_ce);
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Executed operation was terminated, reinitialize it again! \n");
  execute_opcode = 1'b0;
  disable Timeout_execute_opcode;
end

always @ (notify_Twh2_ce) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Minimun nCE positive pulse width time Twh2_ce=%0dns \n", Twh2_ce);
  viol_Twh2_ce = 1'b1;
end

always @ (notify_T_sck_rd1) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Maximum SCK frequency for read operation is 30MHz \n");
  viol_T_sck_rd1 = 1;
end

always @ (notify_T_sck_rd2) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Maximum SCK frequency for read operation with code 0x03 is 15MHz \n");
  viol_T_sck_rd2 = 1;
end

always @ (notify_T_sck_min) begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Maximum SCK frequency is 50MHz \n");
  viol_T_sck_min = 1;
end

always @ (negedge nCE_IN) begin
  viol_Ts1_ce = 0;
  viol_Th1_ce = 0;
end 

always @ (posedge nCE_IN) begin
  viol_T_sck_rd1 = 0;
  viol_T_sck_rd2 = 0;
  viol_T_sck_min = 0;
  viol_Twh_sck = 0;
  viol_Twl_sck = 0;
  viol_Ts_sck_si = 0;
  viol_Th_si_sck = 0;
  viol_Twh2_ce = 0; 
end

// Specify section
always @ (posedge nCE_IN) 
  if (check_Twh1_ce | check_Twh2_ce) Tpos_ce_spi = $time;

always @ (negedge nCE_IN) begin
  if (check_Twh1_ce | check_Twh2_ce) Tneg_ce_spi = $time;
  #(0);
  if ( ((Tneg_ce_spi - Tpos_ce_spi) < Twh1_ce) && ((Tneg_ce_spi - Tpos_ce_spi) > 0) && check_Twh1_ce) begin
    notify_Twh1_ce = ~notify_Twh1_ce;
  end

  if ( ((Tneg_ce_spi - Tpos_ce_spi) < Twh2_ce) && ((Tneg_ce_spi - Tpos_ce_spi) > 0) && check_Twh2_ce) begin
    notify_Twh2_ce = ~notify_Twh2_ce;
  end
end
  
specify
  $period (posedge SCK_IN&&&check_T_sck_min, T_sck_min, notify_T_sck_min);
  $period (posedge SCK_IN&&&check_T_sck_rd1, T_sck_rd1, notify_T_sck_rd1);
  $period (posedge SCK_IN&&&check_T_sck_rd2, T_sck_rd2, notify_T_sck_rd2);

  $width (posedge SCK_IN&&&check_Twh_sck, Twh_sck, 0, notify_Twh_sck);
  $width (negedge SCK_IN&&&check_Twl_sck, Twl_sck, 0, notify_Twl_sck);

  $setup (SI_IN, posedge SCK_IN&&&check_Ts_sck_si, Ts_sck_si, notify_Ts_sck_si);
  $hold (posedge SCK_IN&&&check_Th_si_sck, SI_IN, Th_si_sck, notify_Th_si_sck);

  $setup (negedge nCE_IN, posedge SCK_IN&&&check_Ts1_ce, Ts1_ce, notify_Ts1_ce);
  $hold (posedge SCK&&&check_Th1_ce, posedge nCE_IN, Th1_ce, notify_Th1_ce);

  $setup (posedge nCE_IN, posedge SCK_IN&&&check_Ts2_ce, Ts2_ce, notify_Ts2_ce);
  $hold (posedge SCK_IN&&&check_Th2_ce, negedge nCE_IN, Th2_ce, notify_Th2_ce);
endspecify
// ------------------------------------------------------------------------

assign SEL_SPI_work = SEL_SPI_IN & (~STROBE_IN);
			  
initial begin
  execute_opcode = 1'b0;
  reset_request = 1'b0;
  RDY_BSY_SPI = 1'b0;
  WEL = 1'b0;
  RSTE = 1'b0;
  SPRL = 1'b0;
  EPE = 1'b0;
end

assign SWP = (Sector_Protect_Status_SPI=={numSect{1'b1}}) ? 2'b11 :
			       (Sector_Protect_Status_SPI=={numSect{1'b0}}) ? 2'b00 : 2'b01;

always @ (posedge SCK_IN) if (!RSTN && SEL_SPI_work) if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Behavioral model in power-on reset state. All commmands are ignored \n");

always @ (posedge SCK_IN or posedge nCE_IN or negedge RSTN)
  if (!RSTN) incycle<=1'b0;
  else if (nCE_IN) incycle <=1'b0;
  else incycle <= 1'b1;

always @ (posedge SCK_IN or negedge RSTN)
  if (!RSTN) bit_counter2<=6'h00; 
  else if (!SEL_SPI_work) bit_counter2<=6'h00; 
  else if (!nCE_IN) begin
    if (viol_Ts1_ce || viol_T_sck_min || viol_Twh_sck || viol_Twl_sck || viol_Twh2_ce) bit_counter2<=bit_counter2;
	  else if (bit_counter==6'h00) bit_counter2<=6'h01;
	  else if (((opcode==`OP_RD2) || (opcode==`OP_RDPrReg)) && (bit_counter2==6'h27)) bit_counter2<=6'h20;
	  else if ((opcode==`OP_RD1) && (bit_counter2==6'h2F)) bit_counter2<=6'h28;
	  else if ((opcode==`OP_RDID) && (bit_counter2==6'h17)) bit_counter2<=6'h08;
	  else if ((opcode==`OP_RDStatus) && (bit_counter2==6'h0F)) bit_counter2<=6'h08;
	  else if (bit_counter2<=6'h07) bit_counter2<=bit_counter2 + 1;
	  else if (((opcode==`OP_ER1) || (opcode==`OP_Protect) || (opcode==`OP_Unprotect)) && (bit_counter2<=6'h1F)) bit_counter2<=bit_counter2 + 1;
	  else if (((opcode==`OP_WRStatus) || (opcode==`OP_Reset)) && (bit_counter2<=6'h0F)) bit_counter2<=bit_counter2 + 1; 
	  else if ((opcode==`OP_WR) && (bit_counter2<=6'h27)) bit_counter2<=bit_counter2 + 1;
	  else if ((opcode==`OP_RD2) || (opcode==`OP_RDPrReg) || (opcode==`OP_RDStatus) || (opcode==`OP_RD1) || (opcode==`OP_RDID)) bit_counter2<=bit_counter2 + 1;
  end

always @ (posedge SCK_IN or posedge nCE_IN or negedge RSTN)
  if (!RSTN)	begin
	  bit_counter<=6'h00;
	  opcode<=8'h00;
	  A_SPI<={A_MSB{1'b0}};
	  D_PROG_SPI<=8'h00;
  end else if (nCE_IN) begin
    bit_counter<=6'h00;
  end else if (!SEL_SPI_work) begin 
	  bit_counter<=6'h00;
	  opcode<=8'h00;
	  A_SPI<={A_MSB{1'b0}};
	  D_PROG_SPI<=8'h00;
  end else if (viol_T_sck_min || viol_Twh_sck || viol_Twl_sck || viol_Ts_sck_si || viol_Th_si_sck || 
               viol_Ts1_ce || viol_Twh2_ce) begin
    bit_counter<=6'hx;
	  opcode<=8'hx;
	  A_SPI<={A_MSB{1'bx}};
	  D_PROG_SPI<=8'hx;
  end else begin
    if (bit_counter==6'h00) opcode<={7'h00,SI_IN};
	  else if (bit_counter<=6'h07) opcode<={opcode[6:0],SI_IN};
	  //////////////////
	  if (((opcode==`OP_RD1) || (opcode==`OP_RD2)) && (bit_counter>=6'h08) && RDY_BSY_SPI) begin
      if (Process_ER_Chip) begin
        if (CONSOLE_LOG && EXTRA_LOG) $display("\nTime:%0dns -> ", $time, "Erase Chip operation was terminated by read operation! \n");
        if (CONSOLE_LOG && EXTRA_LOG) $display("\nTime:%0dns -> ", $time, "Command execution result is not guaranteed. Please re-initialize Erase Chip command \n");
        execute_opcode = 1'b0;
        WEL<=1'b0;
        RDY_BSY_SPI<=1'b0;
        Process_ER_Chip<=1'b0;   
        disable Operation_ER_Chip;
      end if (Process_ER_Sector) begin
        if (CONSOLE_LOG && EXTRA_LOG) $display("\nTime:%0dns -> ", $time, "Erase Sector operation was terminated by read operation! \n");
        if (CONSOLE_LOG && EXTRA_LOG) $display("\nTime:%0dns -> ", $time, "Command execution result is not guaranteed. Please re-initialize Erase Sector command \n");
        execute_opcode = 1'b0;
        WEL<=1'b0;
        RDY_BSY_SPI<=1'b0;
        Process_ER_Sector<=1'b0;   
        disable Operation_ER_Sector;
      end if (Process_WR) begin
        if (CONSOLE_LOG && EXTRA_LOG) $display("\nTime:%0dns -> ", $time, "Program operation was terminated by read operation! \n");
        if (CONSOLE_LOG && EXTRA_LOG) $display("\nTime:%0dns -> ", $time, "Command execution result is not guaranteed. Please re-initialize Progran command \n");
        reset_request = 1'b0;
        execute_opcode = 1'b0;
        WEL<=1'b0;
        RDY_BSY_SPI<=1'b0;
        Process_WR<=1'b0;   
        disable Operation_WR;
      end      
    end  
	  //////////////////
	  if ((bit_counter>=6'h08) && (bit_counter<=6'h1F)) A_SPI<={A_SPI[A_MSB-2:0],SI_IN};
	  else if (((opcode==`OP_RD2) && (bit_counter==6'h21)) || ((opcode==`OP_RD1) && (bit_counter==6'h29))) A_SPI<=A_SPI + 1;
	  //////////////////
	  if ((bit_counter>=6'h20) && (bit_counter<=6'h27)) D_PROG_SPI<={D_PROG_SPI[6:0],SI_IN};
	  //////////////////
	  if (((opcode==`OP_RD2) || (opcode==`OP_RDPrReg)) && (bit_counter==6'h27)) bit_counter<=6'h20;
	  else if ((opcode==`OP_RD1) && (bit_counter==6'h2F)) bit_counter<=6'h28;
	  else if ((opcode==`OP_RDID) && (bit_counter==6'h17)) bit_counter<=6'h08;
	  else if ((opcode==`OP_RDStatus) && (bit_counter==6'h0F)) bit_counter<=6'h08;
	  else if (bit_counter<=6'h07) bit_counter<=bit_counter + 1;
	  else if (((opcode==`OP_ER1) || (opcode==`OP_Protect) || (opcode==`OP_Unprotect)) && (bit_counter<=6'h1F)) bit_counter<=bit_counter + 1;
	  else if (((opcode==`OP_WRStatus) || (opcode==`OP_Reset)) && (bit_counter<=6'h0F)) bit_counter<=bit_counter + 1; 
	  else if ((opcode==`OP_WR) && (bit_counter<=6'h27)) bit_counter<=bit_counter + 1;
	  else if ((opcode==`OP_RD2) || (opcode==`OP_RDPrReg) || (opcode==`OP_RDStatus) || (opcode==`OP_RD1) || (opcode==`OP_RDID)) bit_counter<=bit_counter + 1;
  end
	
always @(negedge SCK_IN or posedge nCE_IN or negedge RSTN)
  if (!RSTN) begin
    #(Tpdz_sck_d);
	  SO_OUT <= 1'b0;
	  OE_SO <= 1'b0;
	  data_shift<=8'h00;
  end else if (nCE_IN) begin
    #(Tpdz_sck_d);
	  SO_OUT <= 1'b0;
	  OE_SO <= 1'b0;
	  data_shift<=8'h00;
  end else if (!SEL_SPI_work) begin
    #(Tpdz_sck_d);
	  SO_OUT <= 1'b0;
	  OE_SO <= 1'b0;
	  data_shift<=8'h00;
  end	else begin
    if ((opcode==`OP_RD1) && (bit_counter==6'h28)) begin
      SO_OUT <= 1'bx;
      #(Tpzd_sck_d);
		  if (viol_T_sck_rd1 | viol_Twh_sck | viol_Twl_sck) begin
        if (CONSOLE_LOG && EXTRA_LOG) $display("\nTime:%0dns -> ", $time, "Unable to read data from address: %h \n", A_SPI);
        {SO_OUT, data_shift} <= 8'hx;
		  end else begin 
        if (CONSOLE_LOG && EXTRA_LOG) $display("\nTime:%0dns -> ", $time, "Data: %h read from address: %h \n", ARRAY[A_SPI], A_SPI);
        {SO_OUT, data_shift} <= ARRAY[A_SPI];
      end
		  OE_SO <= 1'b1;
	  end else if ((opcode==`OP_RD1) && (bit_counter>6'h28)) begin
      SO_OUT <= 1'bx;
      #(Tpzd_sck_d);
		  if (viol_T_sck_rd1 | viol_Twh_sck | viol_Twl_sck) begin
        {SO_OUT, data_shift} <= 8'hx;
      end else begin
		    {SO_OUT, data_shift} <= {SO_OUT, data_shift} << 1;
      end
		  OE_SO <= 1'b1;
    end else if ((opcode==`OP_RD2) && (bit_counter==6'h20)) begin
      SO_OUT <= 1'bx;
      #(Tpzd_sck_d);
		  if (viol_T_sck_rd2 | viol_Twh_sck | viol_Twl_sck) begin
        if (CONSOLE_LOG && EXTRA_LOG) $display("\nTime:%0dns -> ", $time, "Unable to read data from address: %h \n", A_SPI);
        {SO_OUT, data_shift} <= 8'hx;
		  end else begin
        if (CONSOLE_LOG && EXTRA_LOG) $display("\nTime:%0dns -> ", $time, "Data: %h read from address: %h \n", ARRAY[A_SPI], A_SPI);
        {SO_OUT, data_shift} <= ARRAY[A_SPI];
      end
		  OE_SO <= 1'b1;
	  end else if ((opcode==`OP_RD2) && (bit_counter>6'h20)) begin
      SO_OUT <= 1'bx;
      #(Tpzd_sck_d);
		  if (viol_T_sck_rd2 | viol_Twh_sck | viol_Twl_sck) begin
        {SO_OUT, data_shift} <= 8'hx;
		  end else begin
		    {SO_OUT, data_shift} <= {SO_OUT, data_shift} << 1;
      end
		  OE_SO <= 1'b1;
	  end	else if ((opcode==`OP_RDStatus) && (bit_counter==6'h08)) begin
      SO_OUT <= 1'bx;
      #(Tpzd_sck_d);
		  if (viol_T_sck_rd1 | viol_Twh_sck | viol_Twl_sck) begin
        if (CONSOLE_LOG && EXTRA_LOG) $display("\nTime:%0dns -> ", $time, "Unable to read status register \n");
        {SO_OUT, data_shift} <= 8'hx;
		  end else begin
        if (CONSOLE_LOG && EXTRA_LOG) $display("\nTime:%0dns -> ", $time, "Read status register");
        if (CONSOLE_LOG && EXTRA_LOG) $display("---> SPRL: %b", SPRL);
        if (CONSOLE_LOG && EXTRA_LOG) $display("---> RSTE: %b", RSTE);
        if (CONSOLE_LOG && EXTRA_LOG) $display("---> EPE: %b", EPE);
        if (CONSOLE_LOG && EXTRA_LOG) $display("---> SWP: %b", SWP);
        if (CONSOLE_LOG && EXTRA_LOG) $display("---> WEL: %b", WEL);
        if (CONSOLE_LOG && EXTRA_LOG) $display("---> RDY_BSY: %b \n", RDY_BSY_SPI);
        {SO_OUT, data_shift}<={SPRL, RSTE, EPE, 1'b0, SWP[1], SWP[0], WEL, RDY_BSY_SPI};
      end
		  OE_SO <= 1'b1;
	  end else if ((opcode==`OP_RDStatus) && (bit_counter>6'h08)) begin
      SO_OUT <= 1'bx;
      #(Tpzd_sck_d);
		  if (viol_T_sck_rd1 | viol_Twh_sck | viol_Twl_sck) begin
        {SO_OUT, data_shift} <= 8'hx;
      end else begin
		    {SO_OUT, data_shift} <= {SO_OUT, data_shift} << 1;
      end
		  OE_SO <= 1'b1;
	  end	else if ((opcode==`OP_RDPrReg) && (bit_counter==6'h20)) begin
      SO_OUT <= 1'bx;
      #(Tpzd_sck_d);
		  if (viol_T_sck_rd1 | viol_Twh_sck | viol_Twl_sck) begin 
        if (CONSOLE_LOG && EXTRA_LOG) $display("\nTime:%0dns -> ", $time, "Unabe to read sector protect register status \n");
        SO_OUT <= 1'bx;
  		end else begin 
        if (CONSOLE_LOG && EXTRA_LOG) $display("\nTime:%0dns -> ", $time, "Read sector protect register status \n");
        if (Sector_Protect_Status_SPI[A_SPI[A_MSB-1:A_SEC_MSB]]) 
          if (CONSOLE_LOG && EXTRA_LOG) $display("\nTime:%0dns -> ", $time, "Sector %0d is protected \n", A_SPI[A_MSB-1:A_SEC_MSB]);
        else if (!Sector_Protect_Status_SPI[A_SPI[A_MSB-1:A_SEC_MSB]]) 
          if (CONSOLE_LOG && EXTRA_LOG) $display("\nTime:%0dns -> ", $time, "Sector %0d is unprotected \n", A_SPI[A_MSB-1:A_SEC_MSB]);
        SO_OUT <= Sector_Protect_Status_SPI[A_SPI[A_MSB-1:A_SEC_MSB]];
      end
		  OE_SO <= 1'b1;
	  end else if ((opcode==`OP_RDID) && (bit_counter==6'h08)) begin
      SO_OUT <= 1'bx;
      #(Tpzd_sck_d);
		  if (viol_T_sck_rd1 | viol_Twh_sck | viol_Twl_sck) begin
        if (CONSOLE_LOG && EXTRA_LOG) $display("\nTime:%0dns -> ", $time, "Unable to read ID \n");
        {SO_OUT, data_shift} <= 8'hx;
      end else begin
        if (CONSOLE_LOG && EXTRA_LOG) $display("\nTime:%0dns -> ", $time, "Manufacture ID: %h \n", id_manufactory);
        {SO_OUT, data_shift} <= id_manufactory;
      end
		  OE_SO <= 1'b1;
	  end	else if ((opcode==`OP_RDID) && (bit_counter==6'h10)) begin
      SO_OUT <= 1'bx;
      #(Tpzd_sck_d);
		  if (viol_T_sck_rd1 | viol_Twh_sck | viol_Twl_sck) begin
        {SO_OUT, data_shift} <= 8'hx;
      end else begin
        if (CONSOLE_LOG && EXTRA_LOG) $display("\nTime:%0dns -> ", $time, "Device ID: %h \n", id_device);
        {SO_OUT, data_shift} <= id_device;
      end
		  OE_SO <= 1'b1;
	  end	else if ((opcode==`OP_RDID) && (bit_counter>6'h08))	begin
      SO_OUT <= 1'bx;
      #(Tpzd_sck_d);
		  if (viol_T_sck_rd1 | viol_Twh_sck | viol_Twl_sck) begin
        {SO_OUT, data_shift} <= 8'hx;
      end else begin
		    {SO_OUT, data_shift} <= {SO_OUT, data_shift} << 1;
      end
		  OE_SO <= 1'b1;
	  end
  end


always @ (posedge execute_opcode) begin: Timeout_execute_opcode // OPERATION EXECUTION TIMEOUT WINDOW DESCRIPTION
  #(Twh1_ce);
  if (viol_Th1_ce) begin
    if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Operation was not executed, reinitialize it again! \n");  
    execute_opcode = 1'b0;
  end else if ((opcode==`OP_Reset) && !RSTE) begin
    if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "RSTE bit is not set. Abort Reset command \n");
    execute_opcode = 1'b0;
  end else if ((opcode==`OP_Reset) && (A_SPI[7:0]!=8'hD0)) begin
    if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "The Confirmation byte received is wrong %h. Abort Reset command \n", A_SPI[7:0]);
    execute_opcode = 1'b0;
  end else if ((opcode==`OP_Reset) && !RDY_BSY_SPI) begin
    if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "Reset during not BUSY \n");
    execute_opcode = 1'b0;
  end else if (opcode==`OP_Reset) begin
    if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "The Reset signal Issued and Device Entered into Idle State \n");
    reset_request = 1'b1;  
  end else if (RDY_BSY_SPI) begin
		if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "Device is busy. Operation execution is not allowed \n");
    execute_opcode = 1'b0; 
  end else begin
	  if ((opcode==`OP_ER1) && !WEL) begin
      if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "WEL bit not set. Abort Erase Sector command \n");
      execute_opcode = 1'b0;     
	  end else if ((opcode==`OP_ER2) && !WEL) begin
      if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "WEL bit not set. Abort Erase Chip command \n");
      execute_opcode = 1'b0;    
	  end else if ((opcode==`OP_WR) && !WEL) begin
      if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "WEL bit not set. Abort Program byte command \n");
      execute_opcode = 1'b0;
	  end else if ((opcode==`OP_Protect) && !WEL) begin
      if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "WEL bit not set. Abort Protect Sector command \n");
      execute_opcode = 1'b0;
	  end else if ((opcode==`OP_Unprotect) && !WEL) begin
      if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "WEL bit not set. Abort Unprotect Sector command \n");
      execute_opcode = 1'b0;
	  end else if ((opcode==`OP_WRStatus) && !WEL) begin
      if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "WEL bit not set. Abort Write Status Register command \n");
      execute_opcode = 1'b0;
	  end else if ((opcode==`OP_Protect) && SPRL) begin 
      if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "Sector Protection Registers are Locked. Abort Protect Sector command \n");
      execute_opcode = 1'b0;
      WEL<=1'b0;
	  end else if ((opcode==`OP_Unprotect) && SPRL) begin 
      if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "Sector Protection Registers are Locked. Abort Unprotect Sector command \n");
      execute_opcode = 1'b0;
      WEL<=1'b0;
    end else if ((opcode==`OP_WR) && Sector_Protect_Status_SPI[A_SPI[A_MSB-1:A_SEC_MSB]]) begin
      if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "Sector %0d for program address %h is protected. Abort Program byte command \n", A_SPI[A_MSB-1:A_SEC_MSB], A_SPI);
      execute_opcode = 1'b0;
      WEL<=1'b0;
    end else if ((opcode==`OP_ER1) && Sector_Protect_Status_SPI[A_SPI[A_MSB-1:A_SEC_MSB]]) begin
      if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "Sector %0d is protected for erase. Abort Erase Sector command \n", A_SPI[A_MSB-1:A_SEC_MSB]);
      execute_opcode = 1'b0;
      WEL<=1'b0;
    end else if ((opcode==`OP_ER2) && |Sector_Protect_Status_SPI) begin
      if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "Some sectors are protected for erase. Abort Erase Chip command \n");
      execute_opcode = 1'b0;
      WEL<=1'b0;
    end else if (opcode==`OP_WEL) begin
      if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "Write Enable Latch Set \n");
      execute_opcode = 1'b0;
      WEL<=1'b1;
      RDY_BSY_SPI<=1'b0;
    end else if (opcode==`OP_WDL) begin
      if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "Write Enable Latch Reset \n");
      execute_opcode = 1'b0;
      WEL<=1'b0;
      RDY_BSY_SPI<=1'b0;
    end else if (opcode==`OP_Protect) begin
      if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "Sector %0d was protected \n", A_SPI[A_MSB-1:A_SEC_MSB]);
      Sector_Protect_Status_SPI[A_SPI[A_MSB-1:A_SEC_MSB]] = 1'b1;
      execute_opcode = 1'b0;
      WEL<=1'b0;
      RDY_BSY_SPI<=1'b0;
    end else if (opcode==`OP_Unprotect) begin
      if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "Sector %0d was unprotected \n", A_SPI[A_MSB-1:A_SEC_MSB]);
      Sector_Protect_Status_SPI[A_SPI[A_MSB-1:A_SEC_MSB]] = 1'b0;
      execute_opcode = 1'b0;
      WEL<=1'b0;
      RDY_BSY_SPI<=1'b0;    
    end else if (opcode==`OP_WRStatus) begin
      if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "Write Status Register operation completed \n");
      if (CONSOLE_LOG && EXTRA_LOG) $display("\nTime:%0dns -> ", $time, "RSTE bit set %b \n", A_SPI[6]);
      if (CONSOLE_LOG && EXTRA_LOG) $display("\nTime:%0dns -> ", $time, "SPRL bit set %b \n", A_SPI[7]);
      RSTE<=A_SPI[6];
      SPRL<=A_SPI[7];
      execute_opcode = 1'b0;
      WEL<=1'b0;
      RDY_BSY_SPI<=1'b0;  
    end else if (opcode==`OP_WR) begin
      execute_opcode = 1'b0;
      RDY_BSY_SPI<=1'b1;  
      EPE<=1'b0;
      ->Event_WR;
    end else if (opcode==`OP_ER1) begin
      execute_opcode = 1'b0;
      RDY_BSY_SPI<=1'b1;  
      EPE<=1'b0;
      ->Event_ER_Sector;
    end else if (opcode==`OP_ER2) begin
      execute_opcode = 1'b0;
      RDY_BSY_SPI<=1'b1;  
      EPE<=1'b0;
      ->Event_ER_Chip;
    end
  end 
end

always @(negedge RSTN or nCE_IN or SEL_SPI_work)
  if (!RSTN) begin
    execute_opcode = 1'b0;
    disable Timeout_execute_opcode;
  end else if (!SEL_SPI_work) begin
    execute_opcode = 1'b0;
    disable Timeout_execute_opcode;
  end else if (nCE_IN) begin
    if ((opcode==`OP_ER1) && ((bit_counter2[2:0] != 3'b000) || (bit_counter2 < 6'h20))) begin
		  if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "nCE deasserted in non-even byte boundary. Abort Sector Erase command \n");
      execute_opcode = 1'b0;
      disable Timeout_execute_opcode;
    end else if ((opcode==`OP_ER2) && (bit_counter2[2:0] != 3'b000)) begin
		  if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "nCE deasserted in non-even byte boundary. Abort Chip Erase command \n");
      execute_opcode = 1'b0;
      disable Timeout_execute_opcode;
    end else if ((opcode==`OP_WR) && ((bit_counter2[2:0] != 3'b000) || (bit_counter2 < 6'h28))) begin
      if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "nCE deasserted in non-even byte boundary. Abort Byte Program command \n");
      execute_opcode = 1'b0;
      disable Timeout_execute_opcode;
    end else if ((opcode==`OP_WEL) && (bit_counter2[2:0] != 3'b000)) begin
      if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "nCE deasserted at non-even byte boundary.  Abort Write Enable command \n");
      execute_opcode = 1'b0;
      disable Timeout_execute_opcode;
    end else if ((opcode==`OP_WDL) && (bit_counter2[2:0] != 3'b000)) begin
      if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "nCE deasserted at non-even byte boundary.  Abort Write Disable command \n");
      execute_opcode = 1'b0;
      disable Timeout_execute_opcode;
    end else if ((opcode==`OP_Protect) && ((bit_counter2[2:0] != 3'b000) || (bit_counter2 < 6'h20))) begin
      if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "nCE deasserted at non-even byte boundary. Abort Protect Sector command \n");
      execute_opcode = 1'b0;
      disable Timeout_execute_opcode;
    end else if ((opcode==`OP_Unprotect) && ((bit_counter2[2:0] != 3'b000) || (bit_counter2 < 6'h20))) begin
      if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "nCE deasserted at non-even byte boundary. Abort Unprotect Sector command \n");
      execute_opcode = 1'b0;
      disable Timeout_execute_opcode;
    end else if ((opcode==`OP_WRStatus) && ((bit_counter2[2:0] != 3'b000) || (bit_counter2 < 6'h10))) begin
      if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "nCE deasserted at non-even byte boundary. Abort Write Status Register command \n");
      execute_opcode = 1'b0;
      disable Timeout_execute_opcode;
    end else if ((opcode==`OP_Reset) && ((bit_counter2[2:0] != 3'b000) || (bit_counter2 < 6'h10))) begin
      if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "Chip Select deasserted at non-even byte boundary. Abort Reset command \n");
      execute_opcode = 1'b0;
      disable Timeout_execute_opcode;
    end else if ((opcode==`OP_ER1) || (opcode==`OP_ER2) || (opcode==`OP_WR) || (opcode==`OP_WEL) || (opcode==`OP_WDL) ||
         (opcode==`OP_Protect) || (opcode==`OP_Unprotect) || (opcode==`OP_WRStatus) || (opcode==`OP_Reset)) begin
      execute_opcode = 1'b1;
    end
  end

always @ (posedge reset_request) begin
  if (Process_ER_Chip) begin
    #(Treset * us_unit);
    if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "Erase Chip operation was resetted \n");
    reset_request = 1'b0;
    execute_opcode = 1'b0;
    WEL<=1'b0;
    RDY_BSY_SPI<=1'b0;
    Process_ER_Chip<=1'b0;   
    disable Operation_ER_Chip;
  end if (Process_ER_Sector) begin
    #(Treset * us_unit);
    if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "Erase Sector operation was resetted \n");
    reset_request = 1'b0;
    execute_opcode = 1'b0;
    WEL<=1'b0;
    RDY_BSY_SPI<=1'b0;
    Process_ER_Sector<=1'b0;   
    disable Operation_ER_Sector;
  end if (Process_WR) begin
    #(Treset * us_unit);
    if (CONSOLE_LOG) $display("\nTime:%0dns -> ", $time, "Program operation was resetted \n");
    reset_request = 1'b0;
    execute_opcode = 1'b0;
    WEL<=1'b0;
    RDY_BSY_SPI<=1'b0;
    Process_WR<=1'b0;   
    disable Operation_WR;
  end 
end

//============================================================================
//======== Read operation description ======== 
//============================================================================
time Tacc;

initial begin
Status_REG = 0;
end
    
always @ (negedge nCE_IN) begin  // Begin read by nCE
	disable read_mode;
  #0; 
  if (RSTN && SEL_PAR && event_r && (Flag_SPR_Verify || 
     (FSM_state_PAR==`State_read_reset) || (FSM_state_PAR==`State_autoselect) || (FSM_state_PAR==`State_byte_program) ||
     (FSM_state_PAR==`State_sector_erase) || (FSM_state_PAR==`State_page_erase) || (FSM_state_PAR==`State_chip_erase))) begin
    Tacc = (Ta_ce);
    A_RD_PAR = A_IN;
	  ->Event_RD;
    #(Tv_oe_d) D_OUT <= {D_MSB{1'bx}};
	end
end

always @ (negedge nOE_IN) begin  // Begin read by nOE
	disable read_mode;
  #0;  
  if (RSTN && SEL_PAR && event_r && (Flag_SPR_Verify || 
     (FSM_state_PAR==`State_read_reset) || (FSM_state_PAR==`State_autoselect) || (FSM_state_PAR==`State_byte_program) ||
     (FSM_state_PAR==`State_sector_erase) || (FSM_state_PAR==`State_page_erase) || (FSM_state_PAR==`State_chip_erase))) begin
    Tacc = (Ta_oe);
    A_RD_PAR = A_IN;
	  ->Event_RD;
    #(Tv_oe_d) D_OUT <= {D_MSB{1'bx}};
	end
end

always @ (A_IN) begin  // Begin read by address
	disable read_mode;
  #0;  
  if (RSTN && SEL_PAR && event_r && ((FSM_state_PAR==`State_read_reset) || (FSM_state_PAR==`State_autoselect) || Flag_SPR_Verify)) begin
    Tacc = (Ta_a);
    A_RD_PAR = A_IN;
	  ->Event_RD; 
    #(Tv_oe_d) D_OUT <= {D_MSB{1'bx}};
	end
end
    
always @ (posedge nCE_IN or posedge nOE_IN or negedge nWE_IN or negedge SEL_PAR or negedge RSTN) begin // End read
  disable read_mode;
  #(Tv_oe_d) D_OUT <= {D_MSB{1'bx}};
end
    
always @ (Event_RD) begin:read_mode
  #(Tacc);
  if (viol_Th_oeh_weh | viol_Tcyr) begin
    viol_Th_oeh_weh = 1'b0;
    viol_Tcyr = 1'b0;
  end else begin
    ->Event_OUT;
  end
end

always @ (Event_OUT) begin
  if (RSTN && SEL_PAR && nWE) begin
    if ((FSM_state_PAR==`State_autoselect) || Flag_SPR_Verify) begin
      case (A_RD_PAR[1:0])
        2'b00: D_OUT <= id_manufactory;
        2'b01: D_OUT <= id_device;
        2'b10: D_OUT <= {7'h00, Sector_Protect_Status_PAR[A_RD_PAR[A_MSB-1:A_SEC_MSB]]};
        2'b11: D_OUT <= 8'h00;
      endcase
    end else if (FSM_state_PAR==`State_read_reset) begin
      if (CONSOLE_LOG && EXTRA_LOG) $display("\nTime:%0dns -> ", $time, "Data: %h read from address: %h \n", ARRAY[A_RD_PAR], A_RD_PAR);
      D_OUT <=  ARRAY[A_RD_PAR];
	  end else if (FSM_state_PAR==`State_byte_program) begin
      Status_REG = {~D_PROG_PAR[7], DQ6, DQ5_write, 5'b00000};
      D_OUT <=  Status_REG;
    end else if ((FSM_state_PAR==`State_sector_erase) || (FSM_state_PAR==`State_page_erase) || (FSM_state_PAR==`State_chip_erase)) begin
      Status_REG = {1'b0, DQ6, DQ5_erase, 1'b0, DQ3, DQ2, 2'b00};
      D_OUT <=  Status_REG;
    end 
	end
end

//============================================================================
//======== Internal operation description ======== 
//============================================================================
initial begin
  Process_WR = 1'b0;
  Process_ER_Page = 1'b0;
  Process_ER_Sector = 1'b0;
  Process_ER_Chip = 1'b0;
  Process_WR_tst = 1'b0;
  Process_ER_Sector_tst = 1'b0;
  Process_SPR = 1'b0;
  Process_SUPR = 1'b0;  
  Load_ER_Sector = 1'b0;
  DQ3 = 1'b0;
  DQ5_write = 1'b0;
  DQ5_erase = 1'b0;
  DQ6 = 1'b0;
	DQ2 = 1'b0;
end

always @ (Event_WR)             Operation_WR;
always @ (Event_ER_Page)        Operation_ER_Page;
always @ (Event_ER_Sector)      Operation_ER_Sector;
always @ (Event_ER_Chip)        Operation_ER_Chip;

always @ (Event_WR_tst) begin: Timeout_WR
  cnt_tck = 0;
  @(posedge TCK_IN) begin cnt_tck = cnt_tck + 1; end
  #(0);
  @(posedge TCK_IN) begin cnt_tck = cnt_tck + 1; end
  #(0);
  @(posedge TCK_IN) begin cnt_tck = cnt_tck + 1; end
  #(0);
  if (cnt_tck == 3) Operation_WR_tst;
end

always @ (Event_ER_Sector_tst) begin: Timeout_ER
  cnt_tck = 0;
  @(posedge TCK_IN) begin cnt_tck = cnt_tck + 1; end
  #(0);
  @(posedge TCK_IN) begin cnt_tck = cnt_tck + 1; end
  #(0);
  @(posedge TCK_IN) begin cnt_tck = cnt_tck + 1; end
  #(0);
  if (cnt_tck == 3) Operation_ER_Sector_tst;
end

always @ (posedge viol_T_tck_wr_min or posedge viol_T_tck_wr_max or posedge viol_T_tck_er_min or posedge viol_T_tck_er_max) 
  if (Process_WR_tst) begin
    Process_WR_tst = 1'b0;
    RDY_BSY_SERIAL = 1'b0;  
    if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Program operation finish unsuccessfully \n"); 
    disable Operation_WR_tst;
    viol_T_tck_wr_min = 1'b0;
    viol_T_tck_wr_max = 1'b0;    
  end else if (Process_ER_Sector_tst) begin
    Process_ER_Sector_tst = 1'b0;
    RDY_BSY_SERIAL = 1'b0; 
    if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Erase Sector operation finish unsuccessfully \n");
    disable Operation_ER_Sector_tst; 
    viol_T_tck_er_min = 1'b0;
    viol_T_tck_er_max = 1'b0;  
  end

// => SECTOR PROTECTION MECHANISM DESCRIPTION BEGIN
initial begin
  #(power_on_reset);
  Sector_Protect_Status_PAR <= {numSect{1'b0}};
  Sector_Protect_Status_SPI <= {numSect{1'b1}};
end

assign reset_protect = ~(A9_HV_IN || OE_HV_IN);

assign Flag_SPR_Verify = A9_HV_IN && (~OE_HV_IN) && ~nCE_IN && nWE_IN && ~nOE_IN;
assign Flag_SPR    = A9_HV_IN && OE_HV_IN && (~(nCE_IN || nWE_IN || A_IN[6]));
assign Flag_SUPR  = A9_HV_IN && OE_HV_IN && (~(nCE_IN || nWE_IN)) && A_IN[6];

always @ * if (Flag_SPR) ->Event_SPR;
always @ * if (Flag_SUPR) ->Event_SUPR;

always @ (Event_SPR) Operation_SPR;
always @ (Event_SUPR) Operation_SUPR;

always @ (negedge Flag_SPR) 
  if (Process_SPR) begin
    Process_SPR = 1'b0;
    Sector_Protect_Status_PAR[A_DLY[A_MSB-1:A_SEC_MSB]] <= 1'bx;
    if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Sector protect operation finish unsuccesfully \n");  
    if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Control signals was deasserted to soon \n"); 
    if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Required minimum timeout %dus for operation pass \n", Tw_we_pr); 
    disable Operation_SPR;  
  end

always @ (negedge Flag_SUPR) 
  if (Process_SUPR) begin
    Process_SUPR = 1'b0;
    Sector_Protect_Status_PAR[A_DLY[A_MSB-1:A_SEC_MSB]] <= 1'bx;
    if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Sector unprotect operation finish unsuccesfully \n");  
    if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Control signals was deasserted to soon \n");  
    if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Timing violation! Required minimum timeout %dms for operation pass \n", Tw_pr_we); 
    disable Operation_SUPR;
  end

always @ (posedge viol_Ts_oe_pr or posedge viol_Ts_a9_pr) begin
  #(0);
  if (Process_SPR) begin
    Process_SPR = 1'b0;
    disable Operation_SPR;
    Sector_Protect_Status_PAR[A_DLY[A_MSB-1:A_SEC_MSB]] <= 1'bx;
    if (viol_Ts_oe_pr) viol_Ts_oe_pr = 0;
    if (viol_Ts_a9_pr) viol_Ts_a9_pr = 0;
    if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Sector protect operation fail for Sector %0d \n", A_DLY[A_MSB-1:A_SEC_MSB]);  
  end else if (Process_SUPR) begin
    Process_SUPR = 1'b0;
    disable Operation_SUPR;
    Sector_Protect_Status_PAR[A_DLY[A_MSB-1:A_SEC_MSB]] <= 1'bx; 
    if (viol_Ts_oe_pr) viol_Ts_oe_pr = 0;
    if (viol_Ts_a9_pr) viol_Ts_a9_pr = 0;
    if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Sector unprotect operation fail for Sector %0d \n", A_DLY[A_MSB-1:A_SEC_MSB]); 
  end
end

// => SECTOR PROTECT SET OPERATION DESCRIPTION BEGIN
task Operation_SPR; 
integer i;
time OP_Time;
begin
  Process_SPR = 1'b1;

  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Sector protect operation start \n");
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Sector %0d selected for protect \n", A_DLY[A_MSB-1:A_SEC_MSB]);   

  Sector_Protect_Status_PAR[A_DLY[A_MSB-1:A_SEC_MSB]] <= 1'bx;

  for (OP_Time = 0; OP_Time < SPR_time; OP_Time = OP_Time + 100) begin
    #(100); 
  end 

  Process_SPR = 1'b0;  

  Sector_Protect_Status_PAR[A_DLY[A_MSB-1:A_SEC_MSB]] <= 1'b1;

  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Sector protect operation finish succesfully \n"); 
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Sector %0d was protected \n", A_DLY[A_MSB-1:A_SEC_MSB]);   
end
endtask 
// <= SECTOR PROTECT SET OPERATION DESCRIPTION END

// => SECTOR UNPROTECT OPERATION DESCRIPTION BEGIN
task Operation_SUPR; 
integer i;
time OP_Time;
begin
  Process_SUPR = 1'b1;

  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Sector unprotect operation start \n");
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Sector %0d selected for unprotect \n", A_DLY[A_MSB-1:A_SEC_MSB]);    

  Sector_Protect_Status_PAR[A_DLY[A_MSB-1:A_SEC_MSB]] <= 1'bx;

  for (OP_Time = 0; OP_Time < SUPR_time; OP_Time = OP_Time + 100) begin
    #(100); 
  end 
  
  Process_SUPR = 1'b0; 

  Sector_Protect_Status_PAR[A_DLY[A_MSB-1:A_SEC_MSB]] <= 1'b0;

  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Sector unprotect operation finish succesfully \n"); 
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Sector %0d was unprotected \n", A_DLY[A_MSB-1:A_SEC_MSB]);   
end
endtask 
// <= SECTOR PROTECT UNSET OPERATION DESCRIPTION END

// <= SECTOR PROTECTION MECHANISM DESCRIPTION END


// => ERASE CHIP OPERATION DESCRIPTION BEGIN
task Operation_ER_Chip; 
integer i, j, k;
time OP_Time;
time Tstart;

reg [A_MSB-A_SEC_MSB-1:0] SA;
begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Erase Chip operation start \n");
  Process_ER_Chip = 1'b1;

  Tstart = 3 * us_unit;
  #(Tstart);

  if ((Sector_Protect_Status_PAR == {numSect{1'b1}}) && SEL_PAR) begin
    if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "All sectors are protected for erase/program operation \n");
    if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Erase Chip operation finish unsuccessfully \n");
    Process_ER_Chip = 1'b0;

    //---------------
    FSM_state_PAR<=`State_read_reset;
    CMD_cycle_PAR<=`Null_cycle;
	  sector_pointer<=0;
	  for (i = 0; i < numSect; i = i + 1) sector_buf[i] <= 0;  
	  A_PROG_PAR<={A_MSB{1'b0}};
	  D_PROG_PAR<=8'h00;
  end else begin

    for (i = 0; i < numSect ; i = i + 1 ) begin
      if (Sector_Protect_Status_PAR[i] && SEL_PAR) if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Sector %0d is protected for erase/program operation \n", i);
    end
    
    SA = 0;
    
    for (i = 0; i < numSect; i = i + 1 ) begin // at first set "unknown state" sectors selected for erase 
      if ((!Sector_Protect_Status_PAR[i] && SEL_PAR) || SEL_SPI) begin
        for (j = 0; j < sizeSector; j = j + 1) begin
          ARRAY[{SA, {A_SEC_MSB{1'b0}}} + j] = {D_MSB{1'bx}};   
        end
      end
      SA = SA + 1;
    end

    for (OP_Time = (ER_Chip_time - Tstart); OP_Time > 0; OP_Time = OP_Time - 100) begin
      #(100);
    end 

    SA = 0;

    for (i = 0; i < numSect; i = i + 1 ) begin // in the end set "one state" sectors selected for erase 
      if ((!Sector_Protect_Status_PAR[i] && SEL_PAR) || SEL_SPI) begin
        for (j = 0; j < sizeSector; j = j + 1) begin
          ARRAY[{SA, {A_SEC_MSB{1'b0}}} + j] = {D_MSB{1'b1}};   
        end
      end
      SA = SA + 1;
    end

    if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Erase Chip operation finish successfully \n");
    Process_ER_Chip = 1'b0;    

    //---------------
    if (SEL_PAR) begin
      FSM_state_PAR<=`State_read_reset;
      CMD_cycle_PAR<=`Null_cycle;
	    sector_pointer<=0;
	    for (i = 0; i < numSect; i=i+1) sector_buf[i] <= 0;  
	    A_PROG_PAR<={A_MSB{1'b0}};
	    D_PROG_PAR<=8'h00;
    //---------------
    end else if (SEL_SPI_work) begin
      RDY_BSY_SPI<=1'b0;  
      WEL<=1'b0;
    end
  end

end
endtask 
// <= ERASE CHIP OPERATION DESCRIPTION END

// => ERASE SECTOR OPERATION DESCRIPTION BEGIN
task Operation_ER_Sector; 
integer i, j, k;
time OP_Time;
time Tstart;

reg [clog2(numSect)-1:0]  sector_buf_temp [numSect-1:0];
reg [clog2(numSect):0]    sector_pointer_temp;

begin
  sector_pointer_temp = 0;
  for (i = 0; i < numSect; i = i + 1 ) sector_buf_temp[i] = 0;

  if (SEL_PAR) begin
    for (i = 0; i < numSect; i = i + 1 ) sector_buf_temp[i] = sector_buf[i];
    sector_pointer_temp = sector_pointer;
  end if (SEL_SPI_work) begin
    sector_buf_temp[0] = A_SPI[A_MSB-1:A_SEC_MSB];
    sector_pointer_temp = 1;
  end

  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Erase Sector operation start \n");
  Process_ER_Sector = 1'b1;

  Tstart = 3 * us_unit;
  #(Tstart);

  for (i = 0; i < sector_pointer_temp; i = i + 1 ) begin
    if (CONSOLE_LOG) $display ("Sector %0d selected for erase \n", sector_buf_temp[i]);    
  end

  k = sector_pointer_temp;
  for (k = 0; k < sector_pointer_temp; k = k + 1 ) begin
    if (Sector_Protect_Status_PAR[sector_buf_temp[k]] && SEL_PAR) begin 
      if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Sector %0d is protected for erase/program operation \n", sector_buf_temp[i]);
      k = k - 1;
    end
  end

  if ((k == 0) && SEL_PAR) begin
    if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "All selected sectors are protected for erase/program operation \n");
    if (CONSOLE_LOG) $display ("Erase Sector operation finish unsuccessfully \n");
    Process_ER_Sector = 1'b0;

    //---------------
    DQ3 = 1'b0;
    FSM_state_PAR<=`State_read_reset;
    CMD_cycle_PAR<=`Null_cycle;
    sector_pointer<=0;
	  for (i = 0; i < numSect; i=i+1) sector_buf[i] <= 0;  
	  A_PROG_PAR<={A_MSB{1'b0}};
	  D_PROG_PAR<=8'h00;
  end else begin

    for (i = 0; i < sector_pointer_temp; i = i + 1 ) begin
      if ((!Sector_Protect_Status_PAR[sector_buf_temp[i]] && SEL_PAR) | SEL_SPI_work) begin
        for (j = 0; j < sizeSector; j = j + 1) begin
          ARRAY[{sector_buf_temp[i], {A_SEC_MSB{1'b0}}} + j] = {D_MSB{1'bx}}; // at first set "unknown state" sectors selected for erase     
        end  

        for (OP_Time = (ER_Sector_time - Tstart); OP_Time > 0; OP_Time = OP_Time - 100) begin
          #(100);
        end
  
        for (j = 0; j < sizeSector; j = j + 1) begin
          ARRAY[{sector_buf_temp[i], {A_SEC_MSB{1'b0}}} + j] = {D_MSB{1'b1}}; // in the end set "one state" sectors selected for erase   
        end
      end
    end

    if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Erase Sector operation finish successfully \n");
    Process_ER_Sector = 1'b0; 

    //---------------
    if (SEL_PAR) begin
      DQ3 = 1'b0;
      FSM_state_PAR<=`State_read_reset;
      CMD_cycle_PAR<=`Null_cycle;
      sector_pointer<=0;
	    for (i = 0; i < numSect; i=i+1) sector_buf[i] <= 0;  
	    A_PROG_PAR<={A_MSB{1'b0}};
	    D_PROG_PAR<=8'h00;  
    //---------------
    end else if (SEL_SPI_work) begin
      RDY_BSY_SPI<=1'b0; 
      WEL<=1'b0;
    end
  end 
end
endtask 
// <= ERASE SECTOR OPERATION DESCRIPTION END

// => ERASE SECTOR TEST OPERATION DESCRIPTION BEGIN
task Operation_ER_Sector_tst; 
integer i, j, k;
time OP_Time;

reg [A_MSB-A_SEC_MSB-1:0] SA;
begin
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Erase Sector operation start \n");
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Sector %0d selected for erase \n", A_SERIAL[A_MSB-1:A_SEC_MSB]);    

  SA = A_SERIAL[A_MSB-1:A_SEC_MSB];

  Process_ER_Sector_tst = #(1) 1'b1;
  for (j = 0; j < sizeSector; j = j + 1) begin
    ARRAY[{SA, {A_SEC_MSB{1'b0}}} + j] = {D_MSB{1'bx}}; // at first set "unknown state" sectors selected for erase    
  end  

  for (OP_Time = ER_Sector_time; OP_Time > 0; OP_Time = OP_Time - 100) begin
    #(100);
  end
  
  for (j = 0; j < sizeSector; j = j + 1) begin
    ARRAY[{SA, {A_SEC_MSB{1'b0}}} + j] = {D_MSB{1'b1}}; // in the end set "one state" sectors selected for erase   
  end

  Process_ER_Sector_tst = #(1) 1'b0; 
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Erase Sector operation finish successfully \n");
  //---------------
  RDY_BSY_SERIAL = 1'b0;
end
endtask 
// <= ERASE SECTOR OPERATION DESCRIPTION END

// => ERASE PAGE OPERATION DESCRIPTION BEGIN
task Operation_ER_Page; 
integer j, k;
time OP_Time;
time Tstart;

reg [A_MSB-A_SEC_MSB-1:0] SA;
reg [clog2(numPage)-1:0] PA;
begin
  SA = A_PROG_PAR[A_MSB-1:A_SEC_MSB];
  PA = A_PROG_PAR[A_SEC_MSB-1:(clog2(numMux)+clog2(numWLP))];

  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Erase Page operation start \n");
  Process_ER_Page = 1'b1;

  Tstart = 3 * us_unit;
  #Tstart;

  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Page %d from Sector %0d selected for erase \n", A_PROG_PAR[A_SEC_MSB-1:11], A_PROG_PAR[A_MSB-1:A_SEC_MSB]);    
  
  if (Sector_Protect_Status_PAR[SA]) begin
    if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Selected page within sector that was protected for erase/program operation \n");
    if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Erase Page operation finish unsuccessfully \n");
    Process_ER_Page = 1'b0;
    //----------------
    FSM_state_PAR<=`State_read_reset;
    CMD_cycle_PAR<=`Null_cycle;
	  sector_pointer<=0;
	  for (i = 0; i < numSect; i=i+1) sector_buf[i] <= 0;  
	  A_PROG_PAR<={A_MSB{1'b0}};
	  D_PROG_PAR<=8'h00;
  end else begin

    for (j = 0; j < sizePage; j = j + 1) begin // at first set "unknown state" page selected for erase    
      ARRAY[{SA, PA, {clog2(numWLP){1'b0}}, {clog2(numMux){1'b0}}} + j] = {D_MSB{1'bx}};   
    end

    for (OP_Time = (ER_Page_time - Tstart); OP_Time > 0; OP_Time = OP_Time - 100) begin
      #(100);
    end 

    for (j = 0; j < sizePage; j = j + 1) begin // in the end set "one state" page selected for erase    
      ARRAY[{SA, PA, {clog2(numWLP){1'b0}}, {clog2(numMux){1'b0}}} + j] = {D_MSB{1'b1}};   
    end

    if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Erase Page operation finish successfully \n");
    Process_ER_Page = 1'b0; 
    //----------------
    FSM_state_PAR<=`State_read_reset;
    CMD_cycle_PAR<=`Null_cycle;
	  sector_pointer<=0;
	  for (i = 0; i < numSect; i=i+1) sector_buf[i] <= 0;  
	  A_PROG_PAR<={A_MSB{1'b0}};
	  D_PROG_PAR<=8'h00;
  end

end
endtask 
// <= ERASE PAGE OPERATION DESCRIPTION END

// => WRITE OPERATION DESCRIPTION BEGIN
task Operation_WR; 
integer j, k;
time OP_Time;
time Tstart;

reg [D_MSB-1:0] original_data;
reg [D_MSB-1:0] data;
reg [A_MSB-1:0] address;
begin

  if (SEL_PAR) begin
    address = A_PROG_PAR;
    data = D_PROG_PAR;
  end else if (SEL_SPI_work) begin
    address = A_SPI;
    data = D_PROG_SPI;
  end 

  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Program operation start \n");
  Process_WR = 1'b1;

  Tstart = 3 * us_unit;
  #(Tstart);

  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Program address: %h / data: %h \n", address, data);    

  if (Sector_Protect_Status_PAR[address[A_MSB-1:A_SEC_MSB]] && SEL_PAR) begin
    if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Program address within sector that was protected for erase/program operation \n");
    if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Program operation finish unsuccessfully \n");
    Process_WR = 1'b0;
    //---------------    
    FSM_state_PAR<=`State_read_reset;
    CMD_cycle_PAR<=`Null_cycle;
  end else begin    
    original_data = ARRAY[address];
    ARRAY[address] = {D_MSB{1'bx}}; // at first set "unknown state" address selected for write 

    for (OP_Time = (WR_time - Tstart); OP_Time > 0; OP_Time = OP_Time - 100) begin
      #(100);
    end 
  
    ARRAY[address] = data & original_data; // in the end set programming data to flash array by AND-ing with original data

    if (ARRAY[address] != data) begin
      if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Program operation finish unsuccessfully \n");
      if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Sector or page with program address must be erased before program operation \n");
      Process_WR = 1'b0; 
      
      //---------------
      if (SEL_PAR) begin
        if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Waiting for Reset command... \n");
        DQ5_write = 1'b1;
      //---------------
      end else if (SEL_SPI_work) begin
        RDY_BSY_SPI<=1'b0;
        EPE<=1'b1;
        WEL<=1'b0; 
      end
    end else begin
      if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Program operation finish successfully \n");
      Process_WR = #(1) 1'b0; 
    
      //---------------
      if (SEL_PAR) begin      
        FSM_state_PAR<=`State_read_reset;
        CMD_cycle_PAR<=`Null_cycle;
      //---------------
      end else if (SEL_SPI_work) begin
        RDY_BSY_SPI<=1'b0; 
        WEL<=1'b0;
      end
    end
  end

end
endtask 
// <= WRITE OPERATION DESCRIPTION END

// => WRITE OPERATION TEST DESCRIPTION BEGIN
task Operation_WR_tst; 
integer j, k;
time OP_Time;
reg [D_MSB-1:0] original_data;
reg [D_MSB-1:0] data;
reg [A_MSB-1:0] address;
begin
  #1;
  address = A_SERIAL;
  data = DIN_SERIAL;

  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Program operation start \n");
  if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Program address: %h / data: %h \n", address, data);    

  Process_WR_tst = #(1) 1'b1;
  original_data = ARRAY[address];
  ARRAY[address] = {D_MSB{1'bx}}; // at first set "unknown state" address selected for write 

  for (OP_Time = WR_time; OP_Time > 0; OP_Time = OP_Time - 100) begin
    #(100);
  end 
  
  ARRAY[address] = data & original_data; // in the end set programming data to flash array by AND-ing with original data

  Process_WR_tst = #(1) 1'b0; 
  RDY_BSY_SERIAL<=1'b0; 

  if (ARRAY[address] != data) begin
    if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Program operation finish unsuccessfully \n");
    if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Sector or page with program address must be erased before program operation \n");
  end else begin
    if (CONSOLE_LOG) $display ("\nTime:%0dns -> ", $time, "Program operation finish successfully \n");
  end
end
endtask 
// <= WRITE OPERATION TEST DESCRIPTION END

endmodule
